::namespace eval ::command {}
::namespace eval ::command::archive {}
array set ::shell::help {
  {archive gunzip} {Decompress a gzipped file}
  {archive zip} {Build a zip archive or kit}
  {archive unzip} {Unpack a zip archive or kit}
  {archive tar} {Build a tarball (tgz)}
  {archive untar} {Unpack a tarball (tgz)}
}
###
# topic: 08d4936a72ed55f4c7b2048a06abe1bfbc19d679
###
proc ::command::archive::gunzip { file {outfile ""} } {
    package require zlib
    # Gunzip the file
    # See http://www.gzip.org/zlib/rfc-gzip.html for gzip file description
    
    set in [open $file r]    
    fconfigure $in -translation binary -buffering none

    set id [read $in 2]
    if { ![string equal $id \x1f\x8b] } {
        error "$file is not a gzip file."
    }
    set cm [read $in 1]
    if { ![string equal $cm \x8] } {
        error "$file: unknown compression method"
    }
    binary scan [read $in 1] b5 FLAGS 
    foreach {FTEXT FHCRC FEXTRA FNAME FCOMMENT} [split $FLAGS ""] {}
    binary scan [read $in 4] i MTIME
    set XFL [read $in 1]
    set OS [read $in 1]

    if { $FEXTRA } {
        binary scan [read $in 2] S XLEN
        set ExtraData [read $in $XLEN]
    }
    set name ""
    if { $FNAME } {    
        set XLEN 1
        set name [read $in $XLEN]
        set c [read $in 1]
        while { $c != "\x0" } {
            append name $c
            set c [read $in 1]
        }
    }
    set comment ""
    if { $FCOMMENT } {
        set c [read $in 1]
        while { $c != "\x0" } {
            append comment $c
            set c [read $in 1]
        }
    }
    set CRC16 ""
    if { $FHCRC } {
        set CRC16 [read $in 2]
    }

    set cdata [read $in]
    close $in

    binary scan [string range $cdata end-7 end] ii CRC32 ISIZE

    set data [zlib inflate [string range $cdata 0 end-8]]
    
    #if { $CRC32 != [zlib crc32 $data] } {
    #    error "gunzip Checksum mismatch."
    #}
    if { $outfile == "" } {
        set outfile $file
        if { [string equal -nocase [file extension $file] ".gz"] } {
            set outfile [file rootname $file]
        }
    }
    if { [string equal $outfile $file] } {
        error "Will not overwrite input file. sorry."
    }
    set out [open $outfile w]
    fconfigure $out -translation binary -buffering none
    puts -nonewline $out $data
    close $out
    file mtime $outfile $MTIME
}

###
# topic: 5619a6e4bf33566da29a8da5d27d4791db37ec47
###
proc ::command::archive::zip args {
  package require zipfile::mkzip
  ::zipfile::mkzip::mkzip {*}$args
}

###
# topic: 7212a7c689c27037fab38448cfe61eaf0fcee07b
###
proc ::command::archive::tar {tarball source} {
  set pwd [pwd]
  cd [file dirname [file normalize $source]] ; doexec tar -czf $tarball [file tail $source]
  cd $pwd  
}

###
# topic: f19813b029c5c92160abf353819c0c40cea75f8f
###
proc ::command::archive::untar {tarball destination} {
  package require tar
  package require zlib

  set pwd [pwd]
  set dpath [file normalize $destination]
  if {![file exists $dpath]} {
    file mkdir $dpath
  }
  cd [file dirname $dpath]
  file delete -force unpack
  file mkdir unpack
  cd unpack
  if {[file extension $tarball] in {.gz .tgz} } {
    ::command::archive::gunzip $tarball [file rootname $tarball]
    set tarball [file rootname $tarball]
  }
  ::tar::untar $tarball
  file delete -force $dpath
  set paths [glob -nocomplain *]
  if {[llength $paths] eq 1 } {
    file rename [lindex $paths 0] $dpath
  } else {
    file mkdir $dpath
    cd [file dirname $dpath]
    file rename unpack $dpath
  }
  cd $pwd
}

###
# topic: 1360d653d7e7fc6f8a103e15d6fcb97a654b50e1
###
proc ::command::archive::unzip {zipfile destination} {
  package require zipfile::decode
  ::zipfile::decode::unzipfile $zipfile $destination
}

namespace eval ::command::archive {
  namespace export *
  namespace ensemble create
}

