
###
# Tools for CC/GCC Environments
###
gort::class create ::gort::tclmake.gcc {
  superclass ::gort::tclmake.generic

  self method match info {
    if { {gcc} ni [dict getnull $info toolset]} {
      return 0
    }
    if {[dict exists $info cc]} {
      set try [dict get $info cc]
    } else {
      if {[dict exists $info cross]} {
        lappend try ${cross}cc ${cross}gcc
      } else {
        lappend try cc gcc
      }
    }
    set exec [::gort::find-EXE CC {*}$try]    
    if {$exec eq ""} {
      return 0
    }
    return 1
  }
  
  ###
  # topic: f3ffeba5470a4b95d9b2fef7c6d834a7d8052d57
  ###
  method initialize {} {
    next
    my variable config
    source [file join ${::gort::home} toolset script_toolset_gcc.tcl]
    ###
    # Translate all "DEFINE" settings to the ENV
    ###
    foreach {var val} [my define-all] {
      set ::env($var) $val
    }
  }

  ###
  # topic: d08f86acb20332c595311dac806474cce32cb2d0
  ###
  method calc-define-output-type {name spec} {
    foreach {type patterns} $spec {
      foreach pattern $patterns {
        if {[string match $pattern $name]} {
          return $type
        }
      }
    }
    return ""
  }
  
 ###
  # topic: 6fb353a784937c19c59e3f3899b5aa67796a5ec7
  # description:
  #    @compile cfile ofile args
  #    
  #    Low level C compiler checker. Compiles and or links a small C program
  #    according to the arguments and returns 1 if OK, or 0 if not.
  #    
  #    Supported settings are:
  #    
  #    -cflags cflags      A list of flags to pass to the compiler
  #    -includes list      A list of includes, e.g. {stdlib.h stdio.h}
  #    -lang c|c++         Use the C (default) or C++ compiler
  #    -libs liblist       List of libraries to link, e.g. {-ldl -lm}
  #    -sourcefile file    Shorthand for -source [readfile [define-get srcdir]/$file]
  ###
  method compile {cfile ofile args} {
    # Easiest way to merge in the settings
    my cc-with $args {
      array set opts [my cc-get-settings]
    }
    set cmdline {}
    lappend cmdline {*}[my define-get CCACHE]
    switch -exact -- $opts(-lang) {
      c++ {
        lappend cmdline {*}[my define-get CXX] {*}[my define-get CXXFLAGS]
      }
      c {
        lappend cmdline {*}[my define-get CC] {*}[my define-get CFLAGS]
      }
      default {
        error "cctest called with unknown language: $opts(-lang)"
      }
    }
    lappend cmdline {*}$opts(-cflags) {*}[my define-get cc-default-debug ""]
    lappend cmdline -c $cfile -o $ofile {*}$opts(-libs)
    exec {*}$cmdline 2>@1
  }
  method link-executable {ofiles execname args} {
    # Easiest way to merge in the settings
    my cc-with $args {
      array set opts [my cc-get-settings]
    }
    set cmdline {}
    lappend cmdline {*}[my define-get CCACHE]
    switch -exact -- $opts(-lang) {
      c++ {
        lappend cmdline {*}[my define-get CXX] {*}[my define-get CXXFLAGS]
      }
      c {
        lappend cmdline {*}[my define-get CC] {*}[my define-get CFLAGS]
      }
      default {
        error "cctest called with unknown language: $opts(-lang)"
      }
    }
    lappend cmdline {*}$opts(-cflags) {*}[my define-get cc-default-debug ""]
    lappend cmdline {*}$ofiles -o $execname {*}$opts(-libs)
    exec {*}$cmdline 2>@1
  }
  

  ###
  # topic: ed7da38a3d95e3b59544ee3fe3afacb4a708fc32
  # description:
  #    Adds the given settings to $::autosetup(ccsettings) and
  #    returns the old settings.
  ###
  method cc-add-settings settings {
    my variable ccsettings
    if {[llength $settings] % 2} {
      error "settings list is missing a value: $settings"
    }
    
    set prev [array get ccsettings]
    array set new $prev
    
    foreach {name value} $settings {
      switch -exact -- $name {
        -cflags - -includes {
          # These are given as lists
          lappend new($name) {*}$value
        }
        -declare {
          lappend new($name) $value
        }
        -libs {
          # Note that new libraries are added before previous libraries
          set new($name) [list {*}$value {*}$new($name)]
        }
        -link - -lang - -nooutput {
          set new($name) $value
        }
        -source - -sourcefile - -code {
          # XXX: These probably are only valid directly from cctest
          set new($name) $value
        }
        default {
          error "unknown cctest setting: $name"
        }
      }
    }
    array set ccsettings [array get new]
    return $prev
  }

  ###
  # topic: 3b3098db8b2939e3678f4ac8f7fcae7b8cef595e
  # description:
  #    @cc-check-decls name ...
  #    
  #    Checks that each given name is either a preprocessor symbol or rvalue
  #    such as an enum. Note that the define used for a decl is HAVE_DECL_xxx
  #    rather than HAVE_xxx
  ###
  method cc-check-decls args {
    set ret 1
    foreach name $args {
      set r [my cctest-decl $name]
      my define-feature "decl $name" $r
      if {!$r} {
        set ret 0
      }
    }
    return $ret
  }

  ###
  # topic: 19b54829e773c755dbf41e19b6727478ed18c9dc
  # description:
  #    @cc-check-defines define ...
  #    
  #    Checks that the given preprocessor symbol is defined
  ###
  method cc-check-defines args {
    my cc-check-some-feature $args {
      my cctest-define $each
    }
  }

  ###
  # topic: 85de85bd314763afd410c27fb18336c3bd577504
  # description:
  #    @cc-check-function-in-lib function libs ?otherlibs?
  #    
  #    Checks that the given given function can be found in one of the libs.
  #    
  #    First checks for no library required, then checks each of the libraries
  #    in turn.
  #    
  #    If the function is found, the feature is defined and lib_$function is defined
  #    to -l$lib where the function was found, or "" if no library required.
  #    In addition, -l$lib is added to the LIBS define.
  #    
  #    If additional libraries may be needed for linking, they should be specified
  #    as $extralibs as "-lotherlib1 -lotherlib2".
  #    These libraries are not automatically added to LIBS.
  #    
  #    Returns 1 if found or 0 if not.
  ###
  method cc-check-function-in-lib {function libs {otherlibs {}}} {
    set found 0
    my cc-with [list -libs $otherlibs] {
      if {[my cctest-function $function]} {
        my define set lib_$function ""
        incr found
      } else {
        foreach lib $libs {
          my cc-with [list -libs -l$lib] {
            if {[my cctest-function $function]} {
              my define set lib_$function -l$lib
              my define set add LIBS -l$lib
              incr found
              break
            }
          }
        }
      }
    }
    if {$found} {
      define [my define-feature-name $function]
    }
    return $found
  }

  ###
  # topic: 2d7e6dca885fac27e61fee985d5f4fb1f95caee9
  # description:
  #    @cc-check-functions function ...
  #    
  #    Checks that the given functions exist (can be linked)
  ###
  method cc-check-functions args {
    my cc-check-some-feature $args {
      my cctest-function $each
    }
  }

  ###
  # topic: 485386ba7f9872f8a92533d5043fb62299b74c3c
  # description:
  #    @cc-check-includes includes ...
  #    
  #    Checks that the given include files can be used
  ###
  method cc-check-includes args {
    my cc-check-some-feature $args {
      set with {}
      if {[dict exists $::autosetup(cc-include-deps) $each]} {
        set deps [dict keys [dict get $::autosetup(cc-include-deps) $each]]
        my cc-check-includes {*}$deps
        foreach i $deps {
          if {[my have-feature $i]} {
            lappend with $i
          }
        }
      }
      if {[llength $with]} {
        my cc-with [list -includes $with] {
          my cctest -includes $each
        }
      } else {
        my cctest -includes $each
      }
    }
  }

  ###
  # topic: b504f8f797677ad6780ad8d7568a9ad561677b45
  # description:
  #    @cc-check-members type.member ...
  #    
  #    Checks that the given type/structure members exist.
  #    A structure member is of the form "struct stat.st_mtime"
  ###
  method cc-check-members args {
    my cc-check-some-feature $args {
      my cctest-member $each
    }
  }

  ###
  # topic: c1442c2c59c2557e2bf16bcffdb586403349cec7
  # description:
  #    @cc-check-progs prog ...
  #    
  #    Checks for existence of the given executables on the path.
  #    
  #    For example, when checking for "grep", the path is searched for
  #    the executable, 'grep', and if found GREP is defined as "grep".
  #    
  #    It the executable is not found, the variable is defined as false.
  #    Returns 1 if all programs were found, or 0 otherwise.
  ###
  method cc-check-progs args {
    set failed 0
    foreach prog $args {
      set PROG [string toupper $prog]
      set exec [::gort::find-EXE $PROG $prog]
      if {$exec eq {}} {
        my define set $PROG false
        incr failed
      } else {
        my define set $PROG $exec
      }
    }
    expr {!$failed}
  }

  ###
  # topic: fff087113039c3de2b468ec7fda531eafe4db0d5
  # description:
  #    @cc-check-sizeof type ...
  #    
  #    Checks the size of the given types (between 1 and 32, inclusive).
  #    Defines a variable with the size determined, or "unknown" otherwise.
  #    e.g. for type 'long long', defines SIZEOF_LONG_LONG.
  #    Returns the size of the last type.
  ###
  method cc-check-sizeof args {
    foreach type $args {
      my msg-checking "Checking for sizeof $type..."
      set size unknown
      # Try the most common sizes first
      foreach i {4 8 1 2 16 32} {
        if {[my cctest -code "static int _x\[sizeof($type) == $i ? 1 : -1\] = { 1 };"]} {
          set size $i
          break
        }
      }
      set define [my define-feature-name $type SIZEOF_]
      my define set $define $size
    }
    # Return the last result
    return $size
  }

  ###
  # topic: 85cfdfef42a6994fc67feb3bc6573f90ce1765b1
  # description:
  #    Checks for each feature in $list by using the given script.
  #    
  #    When the script is evaluated, $each is set to the feature
  #    being checked, and $extra is set to any additional cctest args.
  #    
  #    Returns 1 if all features were found, or 0 otherwise.
  ###
  method cc-check-some-feature {list script} {
    set ret 1
    foreach each $list {
      if {![my check-feature $each $script]} {
        set ret 0
      }
    }
    return $ret
  }

  ###
  # topic: 027f5c4beff37df3bd0a5bba401cf348f0bf5d34
  # description:
  #    @cc-check-tools tool ...
  #    
  #    Checks for existence of the given compiler tools, taking
  #    into account any cross compilation prefix.
  #    
  #    For example, when checking for "ar", first AR is checked on the command
  #    line and then in the environment. If not found, "${host}-ar" or
  #    simply "ar" is assumed depending upon whether cross compiling.
  #    The path is searched for this executable, and if found AR is defined
  #    to the executable name.
  #    Note that even when cross compiling, the simple "ar" is used as a fallback,
  #    but a warning is generated. This is necessary for some toolchains.
  #    
  #    It is an error if the executable is not found.
  ###
  method cc-check-tools args {
    foreach tool $args {
      set TOOL [string toupper $tool]
      set exe [my get-env $TOOL [my define-get cross]$tool]
      if {[set exec [::localtool select-executable {*}$exe]] ne {}} {
        my define set $TOOL $exec
        continue
      }
      if {[set exec [::localtool select-executable {*}$tool]] ne {}} {
        my define set $TOOL $exec
        continue
      }
      error "Failed to find $exe"
    }
  }

  ###
  # topic: b74b3dbce819267d5fcb8cc36e3996cebc1abbb7
  # description:
  #    @cc-check-types type ...
  #    
  #    Checks that the types exist.
  ###
  method cc-check-types args {
    my cc-check-some-feature $args {
      my cctest-type $each
    }
  }

  ###
  # topic: 104da57f5c2a78b290b0e22d5550614b96397cc6
  ###
  method cc-get-settings {} {
    my variable ccsettings
    array get ccsettings
  }

  ###
  # topic: b2d3730a5924a4216df3896b301a778fa8e7a21f
  # description:
  #    @cc-include-needs include required ...
  #    
  #    Ensures that when checking for 'include', a check is first
  #    made for each 'required' file, and if found, it is #included
  ###
  method cc-include-needs {file args} {
    foreach depfile $args {
      dict set ::autosetup(cc-include-deps) $file $depfile 1
    }
  }

  ###
  # topic: 98876143f9328ef15408e64412442b3a333bfca2
  ###
  method cc-store-settings new {
    my variable ccsettings
    array set ccsettings $new
  }

  ###
  # topic: 7ad9840e437e7b59cb9cc50e94c5ab112eedf544
  # description:
  #    Similar to cc-add-settings, but each given setting
  #    simply replaces the existing value.
  #    
  #    Returns the previous settings
  ###
  method cc-update-settings args {
      set prev [my cc-get-settings]
      cc-store-settings [dict merge $prev $args]
      return $prev
  }

  ###
  # topic: eecb10f3fe597d960e1b6ab2c65fb835abfe6a5c
  # description:
  #    @cc-with settings ?{ script }?
  #    
  #    Sets the given 'cctest' settings and then runs the tests in 'script'.
  #    Note that settings such as -lang replace the current setting, while
  #    those such as -includes are appended to the existing setting.
  #    
  #    If no script is given, the settings become the default for the remainder
  #    of the auto.def file.
  #    
  #    cc-with {-lang c++} {
  #    # This will check with the C++ compiler
  #    cc-check-types bool
  #    cc-with {-includes signal.h} {
  #    # This will check with the C++ compiler, signal.h and any existing includes.
  #    ...
  #    }
  #    # back to just the C++ compiler
  #    }
  #    
  #    The -libs setting is special in that newer values are added *before* earlier ones.
  #    
  #    cc-with {-libs {-lc -lm}} {
  #    cc-with {-libs -ldl} {
  #    cctest -libs -lsocket ...
  #    # libs will be in this order: -lsocket -ldl -lc -lm
  #    }
  #    }
  ###
  method cc-with {settings args} {
    if {[llength $args] == 0} {
      my cc-add-settings $settings
    } elseif {[llength $args] > 1} {
      error "usage: cc-with settings ?script?"
    } else {
      set save [my cc-add-settings $settings]
      set rc [catch {uplevel 1 [lindex $args 0]} result info]
      my cc-store-settings $save
      if {$rc != 0} {
        return -code [dict get $info -code] $result
      }
      return $result
    }
  }

  ###
  # topic: 6fb353a784937c19c59e3f3899b5aa67796a5ec7
  # description:
  #    @cctest ?settings?
  #    
  #    Low level C compiler checker. Compiles and or links a small C program
  #    according to the arguments and returns 1 if OK, or 0 if not.
  #    
  #    Supported settings are:
  #    
  #    -cflags cflags      A list of flags to pass to the compiler
  #    -includes list      A list of includes, e.g. {stdlib.h stdio.h}
  #    -declare code       Code to declare before main()
  #    -link 1             Don't just compile, link too
  #    -lang c|c++         Use the C (default) or C++ compiler
  #    -libs liblist       List of libraries to link, e.g. {-ldl -lm}
  #    -code code          Code to compile in the body of main()
  #    -source code        Compile a complete program. Ignore -includes, -declare and -code
  #    -sourcefile file    Shorthand for -source [readfile [define-get srcdir]/$file]
  #    -nooutput 1         Treat any compiler output (e.g. a warning) as an error
  #    
  #    Unless -source or -sourcefile is specified, the C program looks like:
  #    
  #    #include <firstinclude>   /* same for remaining includes in the list */
  #    
  #    declare-code              /* any code in -declare, verbatim */
  #    
  #    int main(void) {
  #    code                    /* any code in -code, verbatim */
  #    return 0;
  #    }
  #    
  #    Any failures are recorded in 'config.log'
  ###
  method cctest args {
    set src conftest__.c
    set tmp conftest__
    
    # Easiest way to merge in the settings
    my cc-with $args {
      array set opts [my cc-get-settings]
    }
  
    if {[info exists opts(-sourcefile)]} {
      set fname [file join [my define-get srcdir] $opts(-sourcefile)]
      if {![file exists $fname]} {
        error "can't find $opts(-sourcefile)"
      }
      set opts(-source) [my readfile $fname]
    }
    if {[info exists opts(-source)]} {
      set lines $opts(-source)
    } else {
      foreach i $opts(-includes) {
        if {$opts(-code) ne "" && ![my feature-checked $i]} {
          # Compiling real code with an unchecked header file
          # Quickly (and silently) check for it now

          # Remove all -includes from settings before checking
          set saveopts [my cc-update-settings -includes {}]
          my cc-check-includes $i
          my cc-store-settings $saveopts
        }
        if {$opts(-code) eq "" || [have-feature $i]} {
          lappend source "#include <$i>"
        }
      }
      lappend source {*}$opts(-declare)
      lappend source "int main(void) \{"
      lappend source $opts(-code)
      lappend source "return 0;"
      lappend source "\}"
      set lines [join $source \n]
    }
    
    # Build the command line
    set cmdline {}
    lappend cmdline {*}[my define-get CCACHE]
    switch -exact -- $opts(-lang) {
      c++ {
        lappend cmdline {*}[my define-get CXX] {*}[my define-get CXXFLAGS]
      }
      c {
        lappend cmdline {*}[my define-get CC] {*}[my define-get CFLAGS]
      }
      default {
        error "cctest called with unknown language: $opts(-lang)"
      }
    }
  
    if {!$opts(-link)} {
      set tmp conftest__.o
      lappend cmdline -c
    }
    lappend cmdline {*}$opts(-cflags) {*}[my define-get cc-default-debug ""]
    lappend cmdline $src -o $tmp {*}$opts(-libs)
  
    # At this point we have the complete command line and the
    # complete source to be compiled. Get the result from cache if
    # we can
    my variable cc_cache
    if {[info exists cc_cache($cmdline,$lines)]} {
      my msg-checking "(cached) "
      set ok $cc_cache($cmdline,$lines)
      return $ok
    }
  
    set fout [open $src w]
    puts $fout $lines\n
    close $fout
    set ok 1
    if {$::autosetup(debug)} {
      puts [list TRYING {*}$cmdline]
    }
    set err [catch {exec {*}$cmdline 2>@1} result errinfo]
    if {$err || ($opts(-nooutput) && [string length $result])} {
      my configlog "Failed: [join $cmdline]"
      my configlog $result
      my configlog "============"
      my configlog "The failed code was:"
      my configlog $lines
      my configlog "============"
      set ok 0
    } elseif {$::autosetup(debug)} {
      my configlog "Compiled OK: [join $cmdline]"
      my configlog "============"
      my configlog $lines
      my configlog "============"
    }
    file delete $src
    file delete $tmp
    # cache it
    set cc_cache($cmdline,$lines) $ok
    return $ok
  }

  ###
  # topic: 05664956c0aa94719b16a4f15201844b12d3cea3
  # description:
  #    Checks for the existence of the given name either as
  #    a macro (#define) or an rvalue (such as an enum)
  ###
  method cctest-decl name {
    my cctest -code "#ifndef $name\n(void)$name;\n#endif"
  }

  ###
  # topic: 5cd7f0f6edffd2b7dae111a278675bfeb52e57e0
  # description: Checks for the existence of the given define by compiling
  ###
  method cctest-define name {
    my cctest -code "#ifndef $name\n#error not defined\n#endif"
  }

  ###
  # topic: 2dd86eebf7e50beddbe2715ffc5432323af9e596
  # description: Checks for the existence of the given function by linking
  ###
  method cctest-function function {
    my cctest -link 1 -declare "extern void $function\(void);" -code "$function\();"
  }

  ###
  # topic: d12950f1c0857592d930f9f914ecd0b86ee047fe
  # description:
  #    Checks for the existence of the given type/structure member.
  #    e.g. "struct stat.st_mtime"
  ###
  method cctest-member struct_member {
    lassign [split $struct_member .] struct member
    my cctest -code "static $struct _s; return sizeof(_s.$member);"
  }

  ###
  # topic: f7a8620e6914c8dcdf6ca30485ac51d89abf8cc5
  # description: Checks for the existence of the given type by compiling
  ###
  method cctest-type type {
    my cctest -code "$type _x;"
  }
  
  ###
  # topic: 7b6250e9fb3334ae0fe4b62aab611106849155df
  # description:
  #    @make-config-header outfile ?-auto patternlist? ?-bare patternlist? ?-none patternlist? ?-str patternlist? ...
  #    
  #    Examines all defined variables which match the given patterns
  #    and writes an include file, $file, which defines each of these.
  #    Variables which match '-auto' are output as follows:
  #    - defines which have the value "0" are ignored.
  #    - defines which have integer values are defined as the integer value.
  #    - any other value is defined as a string, e.g. "value"
  #    Variables which match '-bare' are defined as-is.
  #    Variables which match '-str' are defined as a string, e.g. "value"
  #    Variables which match '-none' are omitted.
  #    
  #    Note that order is important. The first pattern which matches is selected
  #    Default behaviour is:
  #    
  #    -bare {SIZEOF_* HAVE_DECL_*} -auto HAVE_* -none *
  #    
  #    If the file would be unchanged, it is not written.
  ###
  method make-config-header {file args} {
    set guard _[string toupper [regsub -all {[^a-zA-Z0-9]} [file tail $file] _]]
    file mkdir [file dirname $file]
    set lines {}
    lappend lines "#ifndef $guard"
    lappend lines "#define $guard"
    
    # Add some defaults
    lappend args -bare {SIZEOF_* HAVE_DECL_*} -auto HAVE_*
    
    foreach {n value} [lsort -dictionary -stride 1 [my define-all]] {
      set type [my calc-define-output-type $n $args]
      switch -exact -- $type {
        -bare {
          # Just output the value unchanged
        }
        -none {
          continue
        }
        -str {
          set value \"[string map [list \\ \\\\ \" \\\"] $value]\"
        }
        -auto {
          # Automatically determine the type
          if {$value eq "0"} {
            lappend lines "/* #undef $n */"
            continue
          }
          if {![string is integer -strict $value]} {
            set value \"[string map [list \\ \\\\ \" \\\"] $value]\"
          }
        }
        "" {
          continue
        }
        default {
          error "Unknown type in make-config-header: $type"
        }
      }
      lappend lines "#define $n $value"
    }
    lappend lines "#endif"
    set buf [join $lines \n]
    my write-if-changed $file $buf {
      my msg-result "Created $file"
    }
  }
}