::namespace eval ::shed {}
::namespace eval ::shed::classes {}
::namespace eval ::shed::objects {}

proc ::shed::import filename {
  set fin [open $filename r]
  set thisline {}
  while {[gets $fin line]>=0} {
    append thisline \n $line
    if {![info complete $thisline]} continue
    set parseline [string trim $thisline]
    set thisline {}
    if {[string length $parseline]==0} continue
    if {[string index $parseline 0] eq "#"} continue
    dict set nodes [lindex $parseline 0] [lindex $parseline 1]
  }
  close $fin
  
  ::shed::db eval {BEGIN TRANSACTION;}
  try {
    foreach uuid [dict keys $nodes] {
      ::shed::db eval "INSERT OR IGNORE INTO entity (uuid) VALUES (:uuid);"
    }
    foreach {uuid info} $nodes {
      foreach {field value} [dict getnull $info entity] {
        set field [string trimright $field :]
        ::shed::db eval "UPDATE entity set $field=:value where uuid=:uuid;"
      }
      foreach {field value} [dict getnull $info meta] {
        set field [string trimright $field :]
        ::shed::db eval "INSERT or REPLACE INTO meta (entity,field,value) VALUES (:uuid,:field,:value);"
      }
    }
  } on error {err errdat} {
    puts "IMPORT OF $filename aborted: $err"
    ::shed::db eval {ROLLBACK;}
  } on ok {err errdat} {
    ::shed::db eval {COMMIT;}
  }
  ::shed::db eval {BEGIN TRANSACTION;}
  # Inject links
  try {
    foreach {uuid info} $nodes {
      foreach {linktype links} $info {
        if {[string index $linktype end] ne "/"} continue
        set linktype [string trimright $linktype {/}]
        foreach link $links {
          ::shed::db eval {
INSERT OR IGNORE INTO entity(uuid) VALUES (:link);
INSERT OR IGNORE INTO link (fromid,toid,linktype) VALUES (:uuid,:link,:linktype);
}
        }
      }
    }
  } on error {err errdat} {
    puts "IMPORT OF $filename aborted: $err"
    ::shed::db eval {ROLLBACK;}
  } on ok {err errdat} {
    ::shed::db eval {COMMIT;}
  }
}

###
# topic: fab3549f08cc0eb9f92a5ca4239b5248bdcd59d0
###
proc ::shed::getindex path {
  if {[string range $path 0 6] eq "http://"} {
    package require http::wget
    set real_url [::http::_followRedirects $path]
    set token [::http::geturl $real_url]
    if {[::http::ncode $token] != "200"} {
      error "DOWNLOAD FAILED"
    }
    set data [::http::data $token]
    ::http::cleanup $token
    return $data
  } else {
    if {[file tail $path] eq "project.shed"} {
      set shedidx $path 
    } elseif {[file exists [file join $path doc project.shed]]} {
      set shedidx [file join $path doc project.shed]
    } elseif {[file exists [file join $path tool project.shed]]} {
      set shedidx [file join $path tool project.shed]
    } else {
      return {}
    }
    return [::fileutil::cat $shedidx]
  }
}

###
# topic: 250ef4c6e5c69aaff3b464910ba49edab4c21f61
###
proc ::shed::open_db {filename {replace 0}} {
  package require sqlite3
  set replace 0
  if {$filename eq ":memory:"} {
    set present 0
  } else {
    if {$replace} {
      if {[file exists $filename]} {
        file delete $filename
      }
    }
    set present [file exists $filename]
  }
  sqlite3 ::shed::db $filename
  if {!$present} {
    ::shed::db eval [schema INDEX]
  } else {
    ::shed::db eval {
PRAGMA foreign_keys = ON;
    }
    set vers [::shed::db onecolumn {select value from config where name='schema-version'}]
    if {[package vcompare $vers $::shed::schema_version] < 0} {
      # Outdated schema
      puts "Schema out of date. Deleting database"
      ::shed::db close
      file delete $::shed::dbfile
      sqlite3 ::shed::db $filename
      ::shed::db eval [schema]
    }
  }
}

###
# topic: 250ef4c6e5c69aaff3b464910ba49edab4c21f61
###
proc ::shed::open_tempdb {} {
  package require sqlite3
  sqlite3 ::shed::db :memory:
  ::shed::db eval [schema INDEX]
}

###
# topic: 165bb8cb1b7eff45261c67c0d135046a285d8864
# description: Language primitives for TOOL shed description files
###
proc ::shed::pkg_scan_tcl file {
  set dat [::fileutil::cat $file]
  if {![regexp "package (provide|require)" $dat]} {
    return
  }
  set result {}
  set thisline {}
  foreach line [split $dat \n] {
    append thisline $line \n
    if {![info complete $thisline]} {
      continue
    }
    set thisline [string trim $thisline]
    if {[catch {
    while {[set i [string first "\;" $thisline]]>0} {
      set stmt [string range $thisline 0 $i-1]
      if { [string range $stmt 0 14] eq "package provide" } {
        lappend result [lrange $stmt 1 end]
      }
      if { [string range $stmt 0 14] eq "package require" } {
        lappend result [lrange $stmt 1 end]          
      }
      set thisline [string range $i+1 end]
    }
    set stmt $thisline
    set thisline {}
    if { [string range $stmt 0 14] eq "package provide" } {
      lappend result [lrange $stmt 1 end]          
    }
    if { [string range $stmt 0 14] eq "package require" } {
      lappend result [lrange $stmt 1 end]          
    }
    } err]} {
      set thisline {}
    }
  }
  return $result
}

proc ::shed::schema {which} {
  switch $which {
    INDEX { return {
PRAGMA foreign_keys = ON;
create table config (
  name string primary key,
  value any
);
insert into config(name,value) VALUES ('schema-class','shed');
insert into config(name,value) VALUES ('schema-version','1.0');

create table entity(
  uuid uuid primary key,
  shed_class string,
  name string,
  version string
);
CREATE index entity_idx_class ON entity (shed_class,name);

create table meta (
  entity uuid,
  field string,
  value value,
  PRIMARY KEY (entity,field),
  FOREIGN KEY(entity) REFERENCES entity(uuid) ON UPDATE CASCADE
);

create table link(
  fromid uuid,
  toid   uuid,
  linktype string,
  FOREIGN KEY(fromid) REFERENCES entity(uuid) ON UPDATE CASCADE,
  FOREIGN KEY(toid)   REFERENCES entity(uuid) ON UPDATE CASCADE
);
CREATE unique index link_idx_unq ON link (linktype,fromid,toid);
CREATE index link_idx_from ON link (linktype,fromid);
CREATE index link_idx_to   ON link (linktype,toid);
}
    }
    JOURNAL {
PRAGMA foreign_keys = ON;
create table config (
  name string primary key,
  value any
);
insert into config(name,value) VALUES ('schema-class','shed-journal');
insert into config(name,value) VALUES ('schema-version','1.0');
CREATE TABLE journal
		(  seconds   INTEGER NOT NULL, -- seconds since the epoch
		   serial    INTEGER NOT NULL, -- serial number per client invocation
		   action    TEXT, --Install,download,uninstall,etc
		   entity    uuid NOT NULL,    -- entity type
		   name      TEXT NOT NULL,    -- entity name
		   version   TEXT NOT NULL,    -- entity version
		   arch      TEXT NOT NULL,    -- entity architecture
		   PRIMARY KEY (seconds, serial)
		   );
    }
  }
}

###
# topic: 36f17bd0a47976f816750ecff6bbf9632bd1e685
###
proc ::shed::teacup_scan file {
  set result {}
  set dat [split [::fileutil::cat $file] \n]
  if {![regexp "# @@ Meta Begin" $dat]} {
    return {}
  }
  set len [llength $dat]
  set state 0
  set ln 0
  foreach line $dat {
    set line [string trim $line]
    incr ln
    if {$line eq "# @@ Meta Begin"} {
      break
    }    
  }
  set line [string trim [lindex $dat $ln]]
  incr ln
  dict set result teacup_type: [lindex $line 1]
  dict set result name: [lindex $line 2]
  dict set result version: [lindex $line 3]
  foreach line [lrange $dat $ln end] {
    set line [string trim $line]
    incr ln
    if {$line eq "# @@ Meta End"} {
      return $result
    }
    if {[string range $line 0 6] ne "# Meta "} {
      continue
    }
    set cmd [lindex $line 2]
    set dat [string trim [string range $line [string length "# Meta $cmd"] end]]
    switch $cmd {
      "description" {
        dict append result $cmd: "$dat\n"
      }
      require {
        dict lappend result $cmd: [lindex $dat 0]
      }
      default {
        dict set result $cmd: $dat
      }
    }
  }
  return $result
}

