gort::class create ::gort::platform.macosx {
  superclass ::gort::platform.unix
  
  meta set platform os macosx
  meta set teapot os macosx
  
  ###
  # topic: 825cd25953c2cc896a96006b7f454e00
  # title: Return pairings of MAC numbers to IP addresses on the local network
  # description: Under macosx, we call the arp command for arp table resolution
  ###
  method network-arp_table {} {
    set result {}
    set dat [exec arp -a]
    foreach line [split $dat \n] {
      set host [lindex $line 0]
      set ip [lindex $line 1]
      set macid [lindex $line 3]
      lappend result $macid [string range $ip 1 end-1]
    }
    return $result
  }
  
  ###
  # topic: 92ebbfa155883ad41c37d3f843392be4
  # title: Return list of broadcast addresses for local networks
  ###
  method network-broadcast_list {} {
    set result {}
    lappend result 127.0.0.1
    foreach {iface info} [my network-interfaces] {
      if {[dict exists $info broadcast:]} {
        lappend result [dict get $info broadcast:]
      }
    }
    return [lsort -unique -dictionary $result]
  }
  
  ###
  # topic: 187cfa1827097c5cdf1c40c656cedfcc
  # description: Return time since booted
  ###
  method cpuinfo args {
    variable cpuinfo
    if {![info exists cpuinfo]} {
      set cpuinfo {}
      dict set cpuinfo machine  [exec sysctl -n hw.machine]
      dict set cpuinfo cpus     [exec sysctl -n hw.ncpu]
      # Normalize to MB
      dict set cpuinfo memory   [expr {[exec sysctl -n hw.memsize] / 1048576}]
      
      dict set cpuinfo vendor   [exec sysctl -n machdep.cpu.vendor]
      dict set cpuinfo brand    [exec sysctl -n machdep.cpu.brand_string]
      
      dict set cpuinfo model    [exec sysctl -n machdep.cpu.model]
      dict set cpuinfo speed    [expr {[exec sysctl -n hw.cpufrequency]/1000000}]
      
      dict set cpuinfo family   [exec sysctl -n machdep.cpu.family]
      dict set cpuinfo stepping [exec sysctl -n machdep.cpu.stepping]
      dict set cpuinfo features [exec sysctl -n machdep.cpu.features]
      dict set cpuinfo diskless []
    }
    if {$args eq "<list>"} {
      return [dict keys $cpuinfo]
    }
    if {[llength $args]==0} {
      return $cpuinfo
    }
    if {[llength $args]==1} {
      return [dict get $cpuinfo [lindex $args 0]]
    }
    set result {}
    foreach item $args {
      if {[dict exists $cpuinfo $item]} {
        dict set result $item [dict get $cpuinfo $item]
      } else {
        dict set result $item {}
      }
    }
    return $result
  }
  
  ###
  # topic: aa8eda4fb59296a1a34d8d600ca54e28
  # description: Dump interfaces
  ###
  method network-interfaces {} {
    set data [exec ifconfig]
    set iface {}
    set result {}
    foreach line [split $data \n] {
      if {[string index $line 0] in {" " "\t"} } {
        # Indented line appends the prior iface
        foreach {field value} $line {
          dict set result $iface [string trimright $field :]: $value
        }
      } else {
        # Non-intended line - new iface
        set iface [lindex $line 0]
      }
    }
    return $result
  }
  
  ###
  # topic: dd2e2c0810cea69909399808f2a68949
  # title: Return a list of unique hardware addresses
  ###
  method hwid_list {} {
    variable cached_data
    set result {}
    if {![info exists cached_data]} {
      if {[catch {exec system_profiler SPHardwareDataType} hwlist]} {
        set cached_data {}
      } else {
        set cached_data $hwlist
        
      }
    }
    set serial {}
    set hwuuid {}
    set result {}
    catch {
    foreach line [split $cached_data \n] {
      if { [lindex $line 0] == "Serial" && [lindex $line 1] == "Number" } {
        set serial [lindex $line end]
      }
      if { [lindex $line 0] == "Hardware" && [lindex $line 1] == "UUID:" } {
        set hwuuid [lindex $line end]
      }
    }
    }
    if { $hwuuid != {} } {
      lappend result 0x[string map {- {}} $hwuuid]
    }
    # Blank serial number?
    if { $serial != {} } {
      set sn [binary scan $serial H* hash]
      lappend result 0x$hash
    }
    if {[llength $result]} {
      return $result
    }
    foreach mac [my network-mac_list] {
      lappend result 0x[string map {: {}} $mac]
    }
    if {[llength $result]} {
      return $result
    }
    return 0x010203040506
  }
  
  ###
  # topic: d2932eb0ea8cc9f6a865c1ab7cdd4572
  # description:
  #    Called on package load to build any static
  #    structures to cache data that would be time
  #    consuming to call on the fly
  ###
  method initialize {} {
    my variable cpuinfo
    unset -nocomplain cpuinfo
    next
  }
  
  ###
  # topic: 417672d3f31b80d749588365af88baf6
  # title: Return list of ip addresses for this computer (primary first)
  ###
  method network-ip_list {} {
    set result {}
    foreach {iface info} [my network-interfaces] {
      if {[dict exists $info inet:]} {
        lappend result [dict get $info inet:]
      }
    }
    ldelete result 127.0.0.1
    return $result
  }
  
  ###
  # topic: ac9d6815d47f60d45930f0c8c8ae8f16
  # title: Return list of mac numbers for this computer (primary first)
  ###
  method network-mac_list {} {
    set result {}
    foreach {iface info} [my network-interfaces] {
      if {[dict exists $info ether:]} {
        lappend result [dict get $info ether:]
      }
    }
    return $result
  }
  
  ###
  # topic: a43b6f42141820e0ba1094840d0f6fc0
  ###
  method network-network_list {} {
    foreach {iface info} [my network-interfaces] {
      if {![dict exists $info inet:]} continue
      if {![dict exists $info netmask:]} continue
      #set mask [::ip::maskToInt $netmask]
      set addr [dict get $info inet:]
      set mask [dict get $info netmask:]
      set addri [::ip::toInteger $addr]
      lappend result [ip::nativeToPrefix [list [expr {$addri & $mask}] $mask] -ipv4]    
    }
    return $result
  }
  
  ###
  # topic: e7db1ae1b5b98a1bb4384f0a4fe81f42
  ###
  method status {} {
    set result {}
    set loaddat [lindex [exec sysctl -n vm.loadavg] 0]
    set cpus [cpuinfo cpus]
    dict set result cpus $cpus
    dict set result load [expr {[lindex $loaddat 0]*100.0/$cpus}]
    dict set result load_average_1 [lindex $loaddat 0]
    dict set result load_average_5 [lindex $loaddat 1]
    dict set result load_average_15 [lindex $loaddat 2]
  
    set total [exec sysctl -n hw.memsize]
    dict set result memory_total [expr {$total / 1048576}]
    set used 0
    foreach {amt} [exec sysctl -n machdep.memmap] {
      incr used $amt
    }
    dict set result memory_free [expr {($total - $used) / 1048576}]
  
    return $result
  }

  ###
  # topic: eb8a738d9222065f206b21654f52490082be2a11
  ###
  method user_data_root appname {
    return [file join $::env(HOME) Library {Application Support} $appname]
  }
  
  method as_teapot_location {} {
    return /Library/tcl/teapot
  }
}


gort::class create ::gort::platform::macosx-universal {
  superclass ::gort::platform.macosx
  meta set platform 64_bit 0
  meta set platform platform macosx-universal
  meta set platform tcl_compat macosx-universal
  meta set platform tk_compat macosx-universal
  meta set teapot platform macosx-universal
  meta set define CFLAGS  {-O2 -DALLOW_EMPTY_EXPAND -arch x86_64 -arch i386 -pipe -fvisibility=hidden   -isysroot /Developer/SDKs/MacOSX10.6.sdk -mmacosx-version-min=10.5}

  meta set tcl config_flags {--enable-corefoundation=no --enable-framework=yes}
  meta set tk  config_flags {--enable-corefoundation=no --enable-framework=yes}

}

gort::class create ::gort::platform::macosx10.5-i386-x86_64 {
  superclass ::gort::platform.macosx
  meta set platform 64_bit 1
  meta set platform platform macosx10.5-i386-x86_64
  meta set teapot platform macosx10.5-i386-x86_64

  meta set define LDFLAGS {-headerpad_max_install_names -Wl,-search_paths_first}
  meta set define CFLAGS  {-O2 -DALLOW_EMPTY_EXPAND -arch i386 -arch x86_64 -pipe -fvisibility=hidden   -isysroot /Developer/SDKs/MacOSX10.6.sdk -mmacosx-version-min=10.5}
  
  meta set platform tcl_compat {macosx-universal macosx10.5-i386-x86_64 macosx10.5_x11-i386-x86_64}
  meta set platform tk_compat macosx10.5-i386-x86_64
  meta set tcl config_flags {--enable-corefoundation=yes  --enable-framework=no}
  meta set tk  config_flags {--enable-corefoundation=yes  --enable-framework=no --enable-aqua=yes}

}

gort::class create ::gort::platform::macosx10.5_x11-i386-x86_64 {
  superclass ::gort::platform::macosx10.5-i386-x86_64
  meta set teapot platform macosx10.5_x11-i386-x86_64
  meta set platform platform macosx10.5_x11-i386-x86_64
  meta set platform platform_tk macosx10.5_x11-i386-x86_64
  meta set platform tcl_compat {macosx-universal macosx10.5-i386-x86_64 macosx10.5_x11-i386-x86_64}
  meta set platform tk_compat macosx10.5_x11-i386-x86_64
  meta set tcl config_flags {--enable-corefoundation=yes  --enable-framework=no}
  meta set tk  config_flags {--enable-corefoundation=yes  --enable-framework=no --enable-aqua=no --x-includes=/opt/X11/include}
}
