
## Verbose exec

proc doexec {args} {
    global TCL_ERROR

    puts $args
    
    set rc [catch {
	eval "exec $args >@ stdout 2>@ stderr"
    } errmsg]

    if {$rc == $TCL_ERROR} {
	error $errmsg
    }
}

## Add pattern and command to pattern tree.
## 'treeRef' is the name reference, in the calling level, to the array 
##           containing the tree data.
## 'pattern' is the pattern to be added
## 'code' is the block of code related to the pattern.
##
## The pattern tree is a tree where more specific glob patterns appear
## lower down in the tree and equally general patterns appear together in the
## same node.
## To test the relationship between two patterns we compare the pattern,
## as a literal string, to the pattern in a tree node -- and vice versa.
## If the literal string glob-matches the pattern in the tree then it goes 
## beneath the existing pattern. If the match works the other way then the
## pattern will appear higher up in the tree than the existing tree pattern.
##
## If neither pattern matches the other, and the added pattern doesn't match 
## any of the node's siblings, then it becomes a new siblings. If they
## match each other both ways then it means they are almost the same patterns
## except possibly that some *s are ?s (or vice verse) in the other pattern. 
## We check for this by counting the amount of ?s and the pattern with more is 
## considered the more specific (and if they have an equal amount they become 
## siblings).
## 
## If a pattern matches exactly the one in the tree then the related code
## is replaced with the one passed as a parameter.
##
## Yes, this may sound complex but is actually quite simple... ;-)
##
## The tree is implemented as an array. Conceptually each node of the tree
## is an array, which appears as a key-value list in an element of the
## parent array. Thus only the root of the tree really is an array and 
## the key-value lists are set to arrays as child nodes are examined.

proc patternTreeAdd {treeRef pattern code} {
    upvar $treeRef tree

    set i 0
    foreach treePattern $tree(patterns) {
	if {[string compare $treePattern $pattern] == 0} {
	    # Patterns were exactly the same

	    return
	}

	if {([string match $treePattern $pattern] && 
	     ![string match $pattern $treePattern]) || 
	    ([string match $treePattern $pattern] && 
	     [string match $pattern $treePattern] && 
	     ([countChars $pattern "?"] > [countChars $treePattern "?"]))} {

	    # Add pattern to child of this node
	    dputs 2 "Add pattern $pattern as child to $treePattern."

	    if {[llength $tree(children,$treePattern)] != 0} {
		# This level has children so go down to next level
		array set childTree $tree(children,$treePattern)
		patternTreeAdd childTree $pattern $code
		set tree(children,$treePattern) [array get childTree]
	    } else {
		# Make this pattern into a new leaf node

		set newChild(children,$pattern) [list]
		lappend newChild(patterns) $pattern
		set newChild(code,$pattern) $code
		set tree(children,$treePattern) [array get newChild]
	    }

	    return
	} elseif {([string match $pattern $treePattern] && 
		   ![string match $treePattern $pattern]) || 
		  ([string match $pattern $treePattern] && 
		   [string match $treePattern $pattern] && 
		   ([countChars $treePattern "?"] > 
		    [countChars $pattern "?"]))} {

	    # Make 'pattern' new pattern at this level and make 'treePattern' 
	    # into child of 'pattern'
	    dputs 2 "Add pattern $pattern as parent to $treePattern."

	    lappend newChild(patterns) $treePattern
	    set newChild(children,$treePattern) $tree(children,$treePattern)
	    set newChild(code,$treePattern) $tree(code,$treePattern)

	    unset tree(children,$treePattern)
	    unset tree(code,$treePattern)

	    # Check if there are any other patterns in this node which
	    # should go under the new pattern aswell.

	    for {incr i} {$i < [llength $tree(pattern)]} {incr i} {
		set tmpTreePattern [lindex $tree(pattern) $i]

		if {[string match $pattern $tmpTreePattern]} {
		    lappend newChild(patterns) $tmpTreePattern
		    set newChild(children,$tmpTreePattern) \
			$tree(children,$tmpTreePattern)
		    set newChild(code,$tmpTreePattern) \
			$tree(code,$tmpTreePattern)

		    unset tree(children,$tmpTreePattern)
		    unset tree(code,$tmpTreePattern)

		    set tree(patterns) [lreplace $tree(patterns) $i $i]
		}
	    }

	    # Insert 'pattern' where 'treePattern' was
	    set tree(patterns) [lreplace $tree(patterns) $i $i $pattern]
	    set tree(children,$pattern) [array get newChild]
	    set tree(code,$pattern) $code

	    return
	}
	incr i
    }

    # If we reach this stage the pattern should be added to the current
    # node with the other patterns.
    dputs 2 "Add pattern $pattern as sibling."

    lappend tree(patterns) $pattern
    set tree(children,$pattern) [list]
    set tree(code,$pattern) $code
    dputs 3 "tree: [array get tree]"
}


## Searched the rule patterns for a match of 'target' and returns a list of 
## commands to be executed for that target, or an empty string if no matching
## pattern was found.

proc patternTreeSearch {treeRef string} {
    upvar $treeRef tree

    # If we find a match do we go down to the children or not?
    set goDown 1

    set r [list]
    foreach pattern $tree(patterns) {
	if {[string match $pattern $string]} {

	    # Check string with child patterns
	    array set subTree $tree(children,$pattern)

	    if {$goDown && [llength $tree(children,$pattern)] != 0 && 
		[llength [set childRes [patternTreeSearch subTree $string]]] 
		!= 0} {

		lappend r $childRes
	    } else {
		# Execute code at this level.
		lappend r $tree(code,$pattern)
	    }
	}
    }

    return $r
}


## Specify a rule. 'pattern' defines a glob pattern for targets to which the 
## rule apply. 'code' is the operation (like in 'target') to execute. 
##
## This code will actually be used as the body of a procedure that receives 
## two arguments:
## 'target'  -- The target for which the pattern applied
## 'pattern'  -- The pattern it matched.
##
## Specific targets created with the [target] command will override any rule 
## operations for that particular target.

proc setRule {pattern code} {
    variable SmakePatternTree

    set newCode "\n"
    append newCode {set targetUpdate 0}
    append newCode "\n$code"

    append newCode "\n"
    # Automatically return update status (set by depend)
    append newCode {return $targetUpdate}

    namespace eval ::patterns [list proc ${pattern} {target} $newCode]

    if {![info exists SmakePatternTree(patterns)]} {
	set SmakePatternTree(patterns) [list]
    }

    patternTreeAdd SmakePatternTree $pattern ::patterns::$pattern
}


## Return the name of the target at a higher level

proc upTarget {} {
#    upvar #0 smakeTargetPath([getModuleName]) targetPath
    global SmakeTargetPath
    
    if {[llength $SmakeTargetPath] == 1} {
	return [list]
    } else {
	return [lindex $SmakeTargetPath [expr [llength $SmakeTargetPath] -2]]
    }
}


## Set code for a specific target

proc target {target code} {
    global SmakeTargets

    # Initialize update status
    # set code \n{set targetUpdate 0}\n$code
    set newCode "\n"
    append newCode {set targetUpdate 0}
    append newCode "\n$code"

    append newCode "\n"
    # Automatically return update status (set by depend)
    append newCode {return $targetUpdate}

    # Build the target procedure in the "smakeTargets" namespace,
    # which is a child of the current one.
    namespace eval ::smakeTargets [list proc $target {target} $newCode]

    set SmakeTargets($target) {}
}


## Get target list. 
## If 'pattern' is specified it returns names of targets that match glob 
## pattern, otherwise it returns all targets.

proc getTargets {{pattern *}} {
    global SmakeTargets

    set r [array names SmakeTargets $pattern]

    #	return [namespace eval smakeTargets {[info procs]}]
    #	set cmds [info commands smakeTargets::$pattern]
    #	foreach {target} $cmds {
    #		lappend r [string range $target [string length "smakeTargets::"] end]
    #	}

    return $r
}


## Checks if a procedure exists. 'proc' can contain a namespace qualifier.
## In that case procExists checks from inside that namespace.

proc procExists {proc} {
    set qualifier [namespace qualifier $proc]
    
    puts "qualifier: $qualifier"
    puts "nc: [namespace current]"

    smakeTargets::all
    append ex {[llength [info procs } $proc {]] ==0}
    puts "ex: $ex"
    namespace eval $qualifier [list if $ex {
	puts "nomatch"
	return 0
    } else {
	puts "match"
	return 1
    }]
}


## Converts a target name to what it would be as a proc. 

proc convertTargetToProc {target} {
    return ::smakeTargets::$target
}


## Checks to see if 'target' was specified

proc targetExists {target} {
    global SmakeTargets

    return [info exists SmakeTargets($target)]
}


## Executes code for target or rule.
## Returns whatever the command does.

proc execTargetCode {cmd target} {
    dputs 1 called 

    #	variable ModPath 
    global SmakeModDirs SmakeTargetPath
    
    #	set oldPath [pwd]
    #	puts "pwd: $oldPath"

    #	if {[info exists ModPath]} {
    #		cd $ModPath
    #	}

    lappend SmakeTargetPath $target
    set update [$cmd $target]
    dputs 3 "update: $update"
    set SmakeTargetPath [lreplace $SmakeTargetPath end end]
    #	cd $oldPath

    return $update
}


## Check dependencies

proc depend {targets op} {
    dputs 1 called
    global SmakePatternTree DefStarRule SmakeTargetPath

    # Get name of target calling the depend
    # 	set thistarget [lindex $smakeTargetPath end]

    set update 0

    foreach target $targets {
	dputs 3 "target: $target"
	if {[targetExists $target]} {
	    # Code exists to deal with this dependency
	    dputs 2 "target exists"
	    set update [expr $update || [execTargetCode \
					     [convertTargetToProc $target]\
					     $target]]
	} else {
	    # Check for rules

	    set cmds [patternTreeSearch SmakePatternTree $target]

	    if {[llength $cmds] == 0} {
		error "do not know how to build $target"
	    } else {
		foreach cmd $cmds {
		    set update [expr $update || [execTargetCode $cmd $target]]
		}
	    }
	}
    }

    if {$update} {
	uplevel 1 $op
	uplevel 1 {set targetUpdate 1}
    }
}


## Get working directory for target. Just returns the current dir for now.

proc getTargetDir {target} {
    return "."
}


## Compile files

proc compile {args} {
    global Compiler COptions
    eval "doexec $Compiler $COptions -c $args"
}


## Link program

proc link {target objs libs} {
    global Linker LOptions

    set opt_libs ""
    foreach lib $libs {
	set opt_libs "$opt_libs -l$lib"
    }

    eval "doexec $Linker $LOptions -o $target $objs $opt_libs"
}


## Less verbose error procedure

proc errorMsg {msg} {
    global Verbose

    puts stderr $msg
    exit -1
}


## This source command sets 
## Parse options from argv and return the rest of the arguments

proc parse_opt {argc argv} {
    global Smakefile Usage Verbose SmakeVersion
    set parsed_args [list]
    
    set i 0
    while {$i < [llength $argv]} {
	set arg [lindex $argv $i]

	if {[string equal $arg "-f"]} {
	    # Read smakefile from specified file
	    
	    if {$argc == [expr $i+1]} {
		puts stderr $Usage
		exit
	    }

	    incr i
	    set Smakefile [lindex $argv $i]
	} elseif {[string equal $arg "-v"]} {
	    # Verbose errors

	    set Verbose true
	} elseif {[string match "--help" $arg]} {
	    # Show usage
	    puts $Usage
	    exit
	} elseif {[string equal $arg "--version"]} {
	    # Show version
	    puts $SmakeVersion
	    exit
	} else {
	    set parsed_args [lappend $parsed_args $arg]
	}

	incr i
    } 
    return $parsed_args
}


set DefStarRule {
    set upTarget [upTarget]

    set upTargetDir [getTargetDir $upTarget]
    set upFile [file join $upTargetDir [namespace tail $upTarget]]

    set targetDir [getTargetDir $target]
    set targetFile [file join $targetDir [namespace tail $target]]

    dputs 4 "pwd: [pwd]"
    dputs 4 "upTarget: $upTarget, target: $target"
    dputsVar local upFile
    dputsVar local targetFile

    if {[file exists $targetFile]} {
	if {[file exists $upFile]} {
	    dputs 2 "upTarget exists as file"
	    if {[file mtime $targetFile] > [file mtime $upFile]} {
		return 1
	    } else {
		return 0
	    }
	} else {
	    dputs 2 "upTarget does not exist as file"
	    return 1
	}
    } else {
	error "do not know how to build $target"
    }
}

# However, if the NoDebug variable is set, replace everything above
# with empty procs to speed things up.

if {![info exist ::env(GORT_DEBUG)] || [string is false $::env(GORT_DEBUG)] } {
    proc dputs {lvl txt} {}
    proc dputsVar {type args} {}
} else {
  package require fishpool.debug 1.2.2
}


# List of targets that lead to the current target. Includes current target
# as last element in list.
set SmakeTargetPath [list]

#set DEBUG 0
set DbgLevel 5

set Compiler cc
set COptions ""

set Linker cc
set LOptions ""

set ModulePath ""
set SmakeVersion 1.100
set Usage {smake [options] [target]
    Options:
    -f  Read this file instead of default Smakefile
    -v   Verbose mode
    --help  Display this message
    --version  Display Smake version
}
set TCL_ERROR 1

package provide smake 1.100
