## Debug printout
## Authors: Kristoffer Lawson, setok@fishpool.fi
## Errors: Procedures use errorCode "Debug"

package provide fishpool.debug 1.2.2

# When dputs is invoked, messages are printed if their level is lower than
# the one specified by DbgLevel
if {![info exists DbgLevel]} {
    set DbgLevel 0
}

# Debug messages are only display if DEBUG is set to 1
if {![info exists DEBUG]} {
    set DEBUG 1
}

namespace eval dbg {}

if {![info exists dbg::showClicks]} {
	set dbg::showClicks 0
}
if {![info exists dbg::clickDigits]} {
	set dbg::clickDigits 5
}

set DbgPrefix(0) {}  ;# Just print it out as it is
set DbgPrefix(1) "*"  ;# High-level functional info
set DbgPrefix(2) "  *"  ;# Low-level functional info
set DbgPrefix(3) "    *"   ;# High-level status info
set DbgPrefix(4) "      *"   ;# Low-level status info
set DbgPrefix(5) "##"    ;# Checking for something that should always work
# Different prefixes for displaying variables and their values
set DbgPrefix(variable,local) "--var--"  
set DbgPrefix(variable,object) "*var*"
set DbgPrefix(variable,parameter) "**var**"  
set DbgPrefix(variable,param) "**var**"  
set DbgPrefix(variable,global) "##var##"  

## Gets time for debug messages

proc dbg::getTime {} {
	if {$dbg::showClicks} {
		set count [string range [clock clicks end-$dbg::clickDigits end]]
		return "($count)"
	} else {
		return ""
	}
}


## If appropriate debug options are set, prints out the given variable
## names and their values.
## 
## Types (given to 'type' parameter').
## "local" a variable with meaning only to the local procedure.
## "object" a variable with meaning to the whole object instance (in OO).
## "global" a variable relevant to the whole application.
## "parameter" a variable which was given as a parameter to the procedure.
##
## 'args' A list of variable names which should be handled at the level of
##        the caller

proc dputsVar {type args} {
    global DbgLevel DEBUG DbgPrefix

    if {![info exists ::dbg::debugVars]} {
	set ::dbg::debugVars $DEBUG
    }

    if {$::dbg::debugVars} {
	foreach var $args {
	    upvar $var varVal
	    if {![info exists varVal]} {
		error "variable does not exist" {} Debug
	    }
	    set count [string range [clock clicks] end-5 end]
	    append varPairs "$var: $varVal, "
	}
	
	set outputLine "[dbg::getTime] $DbgPrefix(variable,$type) "
	if {[info level] == 1 || ($type == "global")} {
	    # Called from the global level.
	    append outputLine ":: $varPairs"
	} else {		    
	    set ns [uplevel 1 namespace current]
	    set proc [namespace tail [lindex [info level -1] 0]]
	    switch -- $type {
		local {
		    append outputLine \
			"${ns}::${proc}: $varPairs"
		}
		parameter {
		    append outputLine \
			"${ns}::${proc}: $varPairs"
		}
		object {
		    append outputLine ${ns}: $varPairs
		}
		defalt {
		    error "Unrecognised variable type"
		}
	    }
	}

	puts $outputLine
    }

    return
}


## Print debug 'txt' at priority level 'lvl'.
## Levels:
## 0  Just print out 'txt' as it is without caring about the level
## 1  High priority functional info
## 2  Low priority functional info
## 3  High priority status info
## 4  Low priority status info
## 5  Checking for something that should always work

proc dputs {lvl txt} {
    global DbgLevel DbgPrefix DEBUG

    if {$DEBUG && ($lvl<=$DbgLevel)} {
	set count [string range [clock clicks] end-5 end]
	if {[info level] == 1} {
	    puts "[dbg::getTime] $DbgPrefix($lvl) global: $txt"
	} else {
	    if {[string match $txt "called"]} {
		# Procedure call
		puts "[dbg::getTime] $DbgPrefix($lvl) \
                          [uplevel 1 namespace current]::[info level -1]: $txt"
	    } else {
		puts "[dbg::getTime] $DbgPrefix($lvl) \
			[lindex [info level -1] 0]: $txt"
	    }
	}
    }
} 


# However, if the NoDebug variable is set, replace everything above
# with empty procs to speed things up.

if {[info exists NoDebug] && $NoDebug==1} {
    proc dputs {lvl txt} {}
    proc dputsVar {type args} {}
}
