###
# Utilities for automating the build process of C extensions
###

# @synopsis:
#
# CTHULHU modules adds autobuild utilities
#

#::namespace eval ::cthulhu {}

###
# title: define which modules the source we are adding contributes to
###
proc cthulhu_config args {
  set ::CTHULHU(config) [dict merge $::CTHULHU(config) $args]
}

###
# topic: 9c0c2d73c1afa8ef83a739c5d01309d0
# title: Signal for a C header to be read with mkhdr
###
proc cthulhu_add_cheader {filename {trace 0}} {
  set hfilename [::fileutil::relative $::CTHULHU(srcdir) $filename]
  if {$hfilename in $::CTHULHU(headers_verbatim)} {
    return
  }
  if {$hfilename ni $::CTHULHU(headers)} {
    lappend ::CTHULHU(headers) $hfilename
    #::cthulhu_read_csourcefile $file
  }
}

###
# topic: c52ea7e1ff44f11f960d99a55e4ab998
# title: Add the contents of a header file verbatim to the internal headers and public headers
###
proc cthulhu_add_cheader_verbatim {filename {trace 0}} {
  set hfilename [::fileutil::relative $::CTHULHU(srcdir) $filename]
  ldelete ::CTHULHU(headers) $hfilename
  if {$hfilename ni $::CTHULHU(headers_verbatim)} {
    lappend ::CTHULHU(headers_verbatim) $hfilename
  }
}

###
# topic: 91e4d7da8dd82d78af41561360deab10
# title: Signal for a C source to be read with mkhdr
###
proc cthulhu_add_csource {filename {cmdconfig {}}} {
  set config [dict merge $::CTHULHU(config) $cmdconfig]
  
  set cfilename [::fileutil::relative $::CTHULHU(srcdir) $filename]
  dict set ::CTHULHU(sources) $cfilename $config
  #if {$cfilename ni $::CTHULHU(sources)} {
  #  lappend ::CTHULHU(sources) $cfilename
  #}
  if {[string is true [dict getnull $config scan]]} {
    if {$cfilename ni $::CTHULHU(scanlist)} {
      lappend ::CTHULHU(scanlist) $cfilename
    }
  }
}

###
# topic: f11da5f705442524715e8f8fe9af5276
# title: Add a path containing C code
###
proc cthulhu_add_directory {here {cmdconfig {}}} {
  puts [list ADD DIRECTORY $here]
  set config [dict merge {
    cthulhu-ignore-hfiles {}
    cthulhu-ignore-cfiles {}
    build-ignore-cfiles {}
    cthulhu-trace-cfiles {}
  } $::CTHULHU(config) $cmdconfig]

  dict with config {}
  
  set here [file normalize $here]
  ###
  # Execute any .tcl scripts in the generic directory
  ###
  foreach file [lsort  [glob -nocomplain [file join $here *.tcl]]] {
    if {[file tail $file] in "shed.tcl pkgIndex.tcl"} continue
    cd $here
    puts [list SOURCE $file]
    uplevel #0 [list source $file]
    puts [list /source $file]
  }
  ###
  # Build a list of all public header files that
  # need to be amalgamated into the publicly exported
  # version
  ###
  foreach file [lsort  [glob -nocomplain [file join $here *.h]]] {
    cthulhu_include_directory $here
    set fname [file tail $file]
    if {${cthulhu-ignore-hfiles} eq "*"} continue
    if { $fname in ${cthulhu-ignore-hfiles} } continue
    if {[string match *_cases.h $fname]} continue
    cthulhu_add_cheader $file
  }
  foreach file [lsort  [glob -nocomplain [file join $here *.c]]] {
    if {[file tail $file] in ${build-ignore-cfiles} } continue
    cthulhu_add_csource $file
  }
  puts [list /ADD DIRECTORY $here]

}

proc cthulhu_include_directory {here} {
   if { $here ni $::CTHULHU(include_paths) } {
      lappend ::CTHULHU(include_paths) $here
    }
}

###
# topic: 1d3a911fd58337df92205759a6d092c3
# title: Add a source file in Tcl that produces a C file
###
proc cthulhu_add_dynamic {csource tclscript} {
  set cfilename [::fileutil::relative $::CTHULHU(srcdir) $csource]
  set tclfilename [::fileutil::relative $::CTHULHU(srcdir) $tclscript]
  dict set ::CTHULHU(sources) $cfilename tclscript $tclfilename
}

###
# topic: d10665e8da4dd0781bb0a9ced5486e40
# title: Add a pure-tcl library
###
proc cthulhu_add_library {here {cmdconfig {}}} {
  set config [dict merge {
    cthulhu-ignore-tclfiles {}
  } $::CTHULHU(config) $cmdconfig]

  dict with config {}
  set here [file normalize $here]
  foreach file [lsort [glob -nocomplain $here/*.tcl]] {
    if {[file tail $file] in ${cthulhu-ignore-tclfiles}} continue
    set libfilename [::fileutil::relative $::CTHULHU(srcdir) $libfilename]
  }
}

###
# topic: ccfe65b26705afc498e08d3004031066
# title: Detect where we need to produce a _cases.h file to automate a C source
###
proc cthulhu_detect_cases filename {
  set cfilename [::fileutil::relative $::CTHULHU(srcdir) $filename]
  set cases [cthulhu_detect_cases_do $filename]
  puts [list CASES $filename $cfilename $cases]
  if {![llength $cases]} return
  set dirname [file dirname $cfilename]
  foreach case $cases {
    lappend result [file join $dirname $case]
  }
  dict set ::CTHULHU(sources) $cfilename cases $result
}

proc cthulhu_detect_cases_put_item {f x} {
  upvar 1 col col
  if {$col==0} {puts -nonewline $f "   "}
  if {$col<2} {
    puts -nonewline $f [format " %-21s" $x]
    incr col
  } else {
    puts $f $x
    set col 0
  }
}

proc cthulhu_detect_cases_finalize {f} {
  upvar 1 col col
  if {$col>0} {puts $f {}}
  set col 0
}

###
# topic: aacfe07625f74f93dada2159f53fca32
###
proc cthulhu_detect_cases_do cfile {
  set dirname [file dirname $cfile]
  set fin [open $cfile r]
  while {[gets $fin line] >= 0} {
    if {[regexp {^ *case *([A-Z]+)_([A-Z0-9_]+):} $line all prefix label]} {
      lappend cases($prefix) $label
    }
  }
  close $fin

  set col 0
  
  foreach prefix [array names cases] {
    set hfile [file join $dirname [string tolower $prefix]_cases.h]
    if {[file exists $hfile] && [file mtime $hfile]>[file mtime $cfile]} continue
    set f [open $hfile w]
    fconfigure $f -translation crlf
    puts $f "/*** Automatically Generated Header File - Do Not Edit ***/"
    puts $f "  const static char *${prefix}_strs\[\] = \173"
    set lx [lsort  $cases($prefix)]
    foreach item $lx {
      cthulhu_detect_cases_put_item $f \"[string tolower $item]\",
    }
    cthulhu_detect_cases_put_item $f 0
    cthulhu_detect_cases_finalize $f
    puts $f "  \175;"
    puts $f "  enum ${prefix}_enum \173"
    foreach name $lx {
      regsub -all {@} $name {} name
      cthulhu_detect_cases_put_item $f ${prefix}_[string toupper $name],
    }
    cthulhu_detect_cases_finalize $f
    puts $f "  \175;"
    puts $f "\
  int index;
  if( objc<2 ){
    Tcl_WrongNumArgs(interp, 1, objv, \"METHOD ?ARG ...?\");
    return TCL_ERROR;
  }
  if( Tcl_GetIndexFromObj(interp, objv\[1\], ${prefix}_strs,\
            \"option\", 0, &index)){
    return TCL_ERROR;
  }
  switch( (enum ${prefix}_enum)index )"
    close $f
  }
  set result {}
  foreach item [array names cases] {
    lappend result [string tolower ${item}_cases.h]
  }
  return $result
}

###
# topic: 41d95037e5a1cab76939150efdef8939
# title: Declare an end to modifications of ::project
# description:
#    This directive is placed after the last set ::CTHULHU(X) Y
#    but before the first ::cthulhu_add_*
###
proc cthulhu_init args {
  set ::CTHULHU(config) [dict merge {
    target pkg
  } $args]
  set ::CTHULHU(cases)  {}
  set ::CTHULHU(sources)  {}
  set ::CTHULHU(headers)  {}
  set ::CTHULHU(scanlist) {}
  set ::CTHULHU(headers_verbatim) {}
}

###
# topic: 17c9931c3ec5ba115efafaaaa3cf61ed
###
proc cthulhu_mk_lib_init.c outfile {
  global project cout
  set cout      [open $outfile w]
  fconfigure $cout -translation crlf
  puts $cout $::CTHULHU(standard_header)
  puts $cout "#include \"$::CTHULHU(h_file_int)\""
  
  puts $cout "
  
  extern int DLLEXPORT ${CTHULHU(init_funct)}( Tcl_Interp *interp ) \{
    Tcl_Namespace *modPtr\;
  "
  puts $cout {
      /* Initialise the stubs tables. */
  #ifdef USE_TCL_STUBS
      if (
          !Tcl_InitStubs(interp, "8.3", 0)
      ) {
          return TCL_ERROR;
      }
  #endif
  }

  foreach module $::CTHULHU(modules) {
    puts $cout "  if(${module}(interp)) return TCL_ERROR\;"
  }
  foreach {nspace cmds} [lsort -stride 2 -dictionary [array get namespace_commands]] {
    puts $cout "
    modPtr=Tcl_FindNamespace(interp,\"$nspace\",NULL,TCL_NAMESPACE_ONLY)\;
    if(!modPtr) {
      modPtr = Tcl_CreateNamespace(interp, \"$nspace\", NULL, NULL);
    }
    "
    foreach {command cfunct} [lsort -stride 2 -dictionary $cmds] {
      puts $cout "  Tcl_CreateObjCommand(interp,\"::${nspace}::${command}\",(Tcl_ObjCmdProc *)$cfunct,NULL,NULL);"
    }
    puts $cout {
    Tcl_CreateEnsemble(interp, modPtr->fullName, modPtr, TCL_ENSEMBLE_PREFIX);
    Tcl_Export(interp, modPtr, "[a-z]*", 1);
    }
  }
  
  puts $cout {
      /* Register the package. */}
  puts $cout "    if (Tcl_PkgProvide(interp, \"${CTHULHU(pkgname)}\", \"${CTHULHU(pkgvers)}\")) return TCL_ERROR\;"
  
  
  puts $cout "  return TCL_OK\;\n\}"
  close $cout
}

###
# topic: 17c9931c3ec5ba115efafaaaa3cf61ed
###
proc cthulhu_mk_app_init.c outfile {
  global project cout
  set cout      [open $outfile w]
  fconfigure $cout -translation crlf
  puts $cout $::CTHULHU(standard_header)
  puts $cout "#include \"$::CTHULHU(h_file_int)\""
  
  puts $cout "
  
  int ${CTHULHU(init_funct)}_static( Tcl_Interp *interp ) \{
    Tcl_Namespace *modPtr\;
  "

  foreach module $::CTHULHU(modules) {
    puts $cout "
  if(${module}(interp)) \{
    return TCL_ERROR\;
  \}
  "
  }
  foreach {nspace cmds} [lsort -stride 2 -dictionary [array get namespace_commands]] {
    puts $cout "
    modPtr=Tcl_FindNamespace(interp,\"$nspace\",NULL,TCL_NAMESPACE_ONLY)\;
    if(!modPtr) {
      modPtr = Tcl_CreateNamespace(interp, \"$nspace\", NULL, NULL);
    }
    "
    foreach {command cfunct} [lsort -stride 2 -dictionary $cmds] {
      puts $cout "  Tcl_CreateObjCommand(interp,\"::${nspace}::${command}\",(Tcl_ObjCmdProc *)$cfunct,NULL,NULL);"
    }
    puts $cout {
    Tcl_CreateEnsemble(interp, modPtr->fullName, modPtr, TCL_ENSEMBLE_PREFIX);
    Tcl_Export(interp, modPtr, "[a-z]*", 1);
    }
  }
  
  puts $cout {
      /* Register the package. */}
  puts $cout "    if (Tcl_PkgProvide(interp, \"${CTHULHU(pkgname)}\", \"${CTHULHU(pkgvers)}\")) return TCL_ERROR\;"
  
  
  puts $cout "  return TCL_OK\;\n\}"
  close $cout
}

proc cthulhu_compile {extra args} {
    global Compiler COptions
    eval "doexec $Compiler $COptions $extra -c $args"
}

###
# Generate targets for SMAKE
###
proc cthulhu_smake_targets {bldpath srcpath} {
  global project
  file mkdir $bldpath
  set pkg_sources {}
  set pkg_objects {}
  foreach {csource} $::CTHULHU(sources) {
    set ofile [file join $bldpath [string trimleft [string map {/ _ .c .o .. {}} $csource] _]]
    set sourcefile [::fileutil::relative $bldpath [file join $srcpath $csource]]
    lappend pkg_sources $sourcefile
    lappend pkg_objects $ofile
    dict set ::CTHULHU(sources) $csource cfile $sourcefile
    dict set ::CTHULHU(sources) $csource ofile $ofile
  }
  set MYINCLUDES {}
  lappend MYINCLUDES -I.
  lappend MYINCLUDES -I$bldpath
  foreach ipath $::CTHULHU(include_paths) {
    lappend MYINCLUDES -I[::fileutil::relative $bldpath [file join $srcpath $ipath]]
  }
  set ::CTHULHU(PKG_OBJECTS) [lsort  $pkg_objects]
  set ::CTHULHU(PKG_SOURCES) [lsort  $pkg_sources]
  
  foreach {csource cinfo} $::CTHULHU(sources) {
    if {[dict exists $cinfo ofile]} {
      set ofile [dict get $cinfo ofile]
    } else {
      set ofile {}
    }
    set cfile [file join $srcpath $csource]
    set hfiles {}
    if {[dict exists $cinfo cases]} {
      package require cthulhu::mktclotps
      foreach hfile [dict get $cinfo cases] {
        target $hfile [list depends {} [list mktclopts $cfile $bldpath]]
        lappend hfiles $hfile
      }
    }
    puts [list OFILE $ofile]
    puts [list CFILE $cfile]
    if {[dict exists $cinfo tclscript]} {
      target $cfile [list depends {} [list tclexec {*}[dict exists $cinfo tclscript]]]
      if {$ofile ne {}} {
        target $ofile [list depends [list {*}$hfiles $cfile] [list cthulhu_compile [list {*}[dict getnull $cinfo extra] {*}$MYINCLUDES] $cfile -o $ofile]
      }
    } else {
      if {$ofile ne {}} {
        target $ofile [list depends $hfiles [list cthulhu_compile [list {*}[dict getnull $cinfo extra] {*}$MYINCLUDES] $cfile -o $ofile]]
      }
    }
  }
}

###
# Generate the section of a Gnu-Makefile
###
proc cthulhu_mk_sources {bldpath srcpath outfile} {
  global project
  file mkdir $bldpath
  set fout [open $outfile w]
  fconfigure $fout -translation crlf
  set pkg_sources {}
  set pkg_objects {}
  foreach {csource} $::CTHULHU(sources) {
    set ofile [file join $bldpath [string trimleft [string map {/ _ .c .o .. {}} $csource] _]]
    set sourcefile [::fileutil::relative $bldpath [file join $srcpath $csource]]
    lappend pkg_sources $sourcefile
    lappend pkg_objects $ofile
    dict set ::CTHULHU(sources) $csource cfile $sourcefile
    dict set ::CTHULHU(sources) $csource ofile $ofile
  }
  set ILINE "MYINCLUDES="
  append ILINE \\\n"  -I."
  foreach ipath $::CTHULHU(include_paths) {
    append ILINE \\\n"  -I[::fileutil::relative $bldpath [file join $srcpath $ipath]]"
  }
  puts $fout $ILINE
  puts $fout {}
  define PKG_OBJECTS [lsort  $pkg_objects]
  define PKG_SOURCES [lsort  $pkg_sources]
  
  #puts $fout "build/$CTHULHU(c_file):"
  #puts $fout "\t\${TCLSH} scripts/cthulhu.tcl\n"
  
  foreach {csource cinfo} $::CTHULHU(sources) {
    if {[dict exists $cinfo ofile]} {
      set ofile [dict get $cinfo ofile]
    } else {
      set ofile {}
    }
    set hfiles {}
    if {[dict exists $cinfo cases]} {
      foreach hfile [dict get $cinfo cases] {
        puts $fout "$hfile:"
        puts $fout "\t\$(TCLSH_PROG) scripts/mktclopts.tcl [dict get cinfo $cfile]\n"
        lappend hfiles $hfile
      }
    }
    if {[dict exists $cinfo tclscript]} {
      set cfile [file rootname $ofile].c
      puts $fout "${cfile}:"
      puts $fout "\t\$(TCLSH_PROG) [file join $srcpath [dict get $cinfo tclscript]] [file join $bldpath $cfile].c\n"
      if {$ofile != {}} {
        puts $fout "$ofile: $hfiles"
        puts $fout "\t\$(COMPILE) [dict getnull $cinfo extra] \$(MYINCLUDES) -c $cfile -o \$@\n"
      }
    } else {
      if {$ofile != {}} {
        puts $fout "$ofile: $hfiles"
        puts $fout "\t\$(COMPILE) [dict getnull $cinfo extra] \$(MYINCLUDES) -c [dict get $cinfo cfile] -o \$@\n"
      }
    }
  }
  close $fout
}

###
# topic: f7eec240dada25d73c1f68a877fa40be
# title: Produce the PROJECT.decls file
# description: Tools for automating the process of building stubs libraries
###
proc cthulhu_mk_stub_decls {pkgname mkhdrfile path} {
  set outfile [file join $path/$pkgname.decls]
  
  ###
  # Build the decls file
  ###
  set fout [open $outfile w]
  puts $fout [subst {###
  # $outfile
  #
  # This file was generated by [info script]
  ###
  
  library $pkgname
  interface $pkgname
  }]
  
  set fin [open $mkhdrfile r]
  set thisline {}
  set functcount 0
  while {[gets $fin line]>=0} {
    append thisline \n $line
    if {![info complete $thisline]} continue
    set readline $thisline
    set thisline {}
    set type [lindex $readline 1]
    if { $type ne "f" } continue
  
    set infodict [lindex $readline end]
    if {![dict exists $infodict definition]} continue
    set def [dict get $infodict definition]
    set def [string trim $def]
    set def [string trimright $def \;]
    if {![string match "*STUB_EXPORT*" $def]} continue
    puts $fout [list declare [incr functcount] $def]
    
  }
  close $fin
  close $fout
  
  ###
  # Build [package]Decls.h
  ###
  set hout [open [file join $path ${pkgname}Decls.h] w]
  
  close $hout

  set cout [open [file join $path ${pkgname}StubInit.c] w]
puts $cout [string map [list %pkgname% $pkgname %PkgName% [string totitle $pkgname]] {
#ifndef USE_TCL_STUBS
#define USE_TCL_STUBS
#endif
#undef USE_TCL_STUB_PROCS

#include "tcl.h"
#include "%pkgname%.h"

 /*
 ** Ensure that Tdom_InitStubs is built as an exported symbol.  The other stub
 ** functions should be built as non-exported symbols.
 */

#undef TCL_STORAGE_CLASS
#define TCL_STORAGE_CLASS DLLEXPORT

%PkgName%Stubs *%pkgname%StubsPtr;

 /*
 **----------------------------------------------------------------------
 **
 **  %PkgName%_InitStubs --
 **
 **        Checks that the correct version of %PkgName% is loaded and that it
 **        supports stubs. It then initialises the stub table pointers.
 **
 **  Results:
 **        The actual version of %PkgName% that satisfies the request, or
 **        NULL to indicate that an error occurred.
 **
 **  Side effects:
 **        Sets the stub table pointers.
 **
 **----------------------------------------------------------------------
 */

char *
%PkgName%_InitStubs (Tcl_Interp *interp, char *version, int exact)
{
  char *actualVersion;
  
  actualVersion = Tcl_PkgRequireEx(interp, "%pkgname%", version, exact,
                                                                   (ClientData *) &%pkgname%StubsPtr);
  if (!actualVersion) {
        return NULL;
  }
  
  if (!%pkgname%StubsPtr) {
        Tcl_SetResult(interp,
                                  "This implementation of %PkgName% does not support stubs",
                                  TCL_STATIC);
        return NULL;
  }
  
  return actualVersion;
}
}]
  close $cout
}

###
# topic: ba1d2c7e8eab96029e434d54f917ef5a
###
proc cthulhu_mkhdr_index {hout docfileout} {
  global project
  set scanlist {}
  foreach file $::CTHULHU(headers) {
    lappend scanlist $file
  }
  foreach file $::CTHULHU(scanlist) {
    lappend scanlist $file
  }
  ldelete scanlist  [::file normalize  [file join $::BUILDDIR $::CTHULHU(h_file_int)]]
  ldelete scanlist  [::file normalize  [file join $::BUILDDIR $::CTHULHU(c_file)]]
  puts "WRITING INTERNAL HEADERS TO $hout"
  set fout [open $hout w]
puts $fout "/*
** DO NOT EDIT THIS FILE
** It is automagically generated by scripts/cthulhu.tcl
*/"
  fconfigure $fout -translation crlf
  foreach file $::CTHULHU(headers_verbatim) {
    puts $fout "/* Verbatim headers */"
    set fullname [file join $::CTHULHU(srcdir) $file]
    set type [file type $fullname]
    if {$type ne "file"} continue
    puts $fout "/*\n *$file \n*/"
    set fin [open $fullname r]
    puts $fout [read $fin]
    close $fin
  }
  puts $fout "/* FUNCTION DECLARATIONS */"
  ###
  # Do get around platform line breaks, we output to a tmp file
  # and concat in Tcl
  ###
  set crscanlist {}
  foreach file $scanlist {
    set crfile $file.cr[file extension $file]
    set rawfin [open $file r]
    set rawfout [open $crfile w]
    fconfigure $rawfout -translation lf
    puts $rawfout [read $rawfin]
    close $rawfout
    close $rawfin
    lappend crscanlist $crfile
  }
  
  ::cthulhu_mkhdr -h -- {*}$crscanlist > $hout.cr
  set rawfin [open $hout.cr r]
  puts $fout [read $rawfin]
  close $rawfin
  file delete $hout.cr
  close $fout
  
  ::cthulhu_mkhdr -doc -- {*}$scanlist > $docfileout
  
  foreach file $crscanlist {
    file delete $file
  }

  set fin [open $hout r]
  while {[gets $fin line]>=0} {
    if {[regexp TCL_MODULE $line] || [regexp DLLEXPORT $line]} {
      foreach regexp {
           {(.*) ([a-zA-Z_][a-zA-Z0-9_]*) *\((.*)\)}
           {(.*) (\x2a[a-zA-Z_][a-zA-Z0-9_]*) *\((.*)\)}
      } {
        if {[regexp $regexp $line all keywords funcname arglist]} {
          lappend ::CTHULHU(modules) $funcname        
          break
        }
      }
    }
  }
}

proc cthulhu_find_mkhdr {} {
  set exename [lindex [find-an-executable mkhdr] 0]
  if {$exename ne {}} {
    return [list exec [::file normalize  $exename]]
  }
  if {[info exists ::odie(mkhdr)]} {
    if {[file exists [::file normalize  $::odie(mkhdr)]]} {
      return [list exec [::file normalize  $::odie(mkhdr)]]
    } 
  }
  doexec $::odie(cc) -o mkhdr.o -c $::odie(src_dir)/scripts/mkhdr.c
  doexec $::odie(cc) mkhdr.o -o mkhdr$::odie(exe_suffix)
  file copy -force mkhdr$::odie(exe_suffix) [::file normalize  $::odie(mkhdr)]
  return [list exec [::file normalize  $::odie(mkhdr)]]
  error {mkhdr not available}
}


proc cthulhu_mkhdr args {
  set cmd [cthulhu_find_mkhdr]
  {*}${cmd} {*}$args
}

array set ::CTHULHU {
  headers_verbatim {}
  include_paths {}
  sources {}
  tcl_sources {}
  modules {}
  srcdir {}
  config {}
}
set BUILDPATH [pwd]

set HERE [file dirname [file normalize [info script]]]
foreach file {debug.tcl smake.tcl} {
  source [file join $HERE $file]
}

package provide cthulhu 2.0