#!/usr/bin/tclsh
#
# This Tcl/Tk script implements a simple text editor.
#
# For the public domain.
#
package require Tk
tk appname tkedit

# Scan for command line arguments
#
set DeleteAfterLoading 0
set Title {}
set ReadOnly 0
set new_argv {}
set forceMulti 0
set AutoWidth 0
foreach arg $argv {
  if {[info exists destination]} {
    set $destination $arg
    unset destination
    continue
  }
  switch -glob -- $arg {
    -autowidth {
       set AutoWidth 1
    }
    -width {
       set destination DesiredWidth
    }
    -goto {
       set destination DesiredGoto
    }
    -force {
       set forceMulti 1
    }
    -remove {
       set DeleteAfterLoading 1
    }
    -title {
       set destination Title
    }
    -readonly {
       set ReadOnly 1
       tk appname tkreader
    }
    -help -
    -* {
       puts "Usage: $argv0 \[options\] \[files\]"
       puts "Options:"
       puts "    -remove      Delete the file after loading"
       puts "    -help        Print this message"
       puts "    -title       Set the title of the editor"
       puts "    -readonly    Do not allow editing"
       exit
    }
    default {
       lappend new_argv $arg
    }
  }
}
set argv $new_argv
set argc [llength $argv]
option add *highlightThickness 0

# The next pair of functions implement a dialog box that tells about
# this program.  The dialog box is the one that appears when the user
# selects Help/AboutThisProgram from the main menu.
#
proc AboutMessage {w text} {
  label $w.title -text TkEdit \
    -font -adobe-times-bold-i-normal--24-240-75-75-p-128-iso8859-1
  pack $w.title -side top -pady 15
  message $w.subtitle -width 10c -justify center \
    -font -adobe-times-bold-i-normal-*-14-140-75-75-p-77-iso8859-1 \
    -text "An ASCII text editor built upon\nthe Tcl/Tk Text widget"
  pack $w.subtitle -side top -pady 10 -padx 15
  message $w.msg -width 10c -text "
By D. Richard Hipp
Hipp, Wyrick & Company, Inc.
6200 Maple Cove Lane
Charlotte, NC 28269
704-948-4565
drh@sqlite.org" \
    -font -adobe-times-medium-r-normal-*-12-120-75-75-p-64-iso8859-1
  pack $w.msg -padx 15 -anchor w
  label $w.spacer -text { }
  pack $w.spacer
}
proc HelpAboutThisProgram {} {
  ModalDialogBox . {About} AboutMessage {} {{Dismiss {}}}
}

# Get the name of the file to be edited.
#
if {$argc>1} {
  if {$argc>5 && !$forceMulti} {
    puts stderr "too many arguments: use -force to spawn them all"
    exit 1
  }
  # If more than one filename is listed on the command line, invoke
  # a separate editor for each one.
  foreach i $argv {
    catch {exec $argv0 $i &}
  }
  exit 0
}
if {$argc==1} {
  # This is the case where a file is specified on the command line.
  # Bring up the editor on that single file
  #
  set path $argv
  if {"[string index $path 0]"!="/"} {
    set path [pwd]/$path
  }
  regsub -all {/\./} $path / path
  while {[regsub -all {[^/]*[^/.]/\.\./} $path {} path]} {}
  regsub {^(\./)+} $path {} path
  set FullPath {}

  # See if any other editor is currently working on this file.  If it is,
  # then deiconify and raise that editor and exit, because we don't need
  # more than a single editor running on each file.
  #
  wm withdraw .
  if {!$ReadOnly} {
    foreach other [winfo interps] {
      if {![string match tkedit* $other]} continue
      # if {![string match tkedit* $other] && $other!="e" 
      #    && ![string match {e *} $other]} {
      #  continue
      #}
      catch {
        if {[send $other {set tkedit_FullPath}]==$path} {
          send $other {wm deiconify .; raise .}
          exit 0
        }
      }
    }
  }

  set FullPath $path
  set tkedit_FullPath $path
  wm deiconify .
} else {
  # This is the case where no filename is given.  Simply start a new
  # editor.
  #
  set FullPath {}
  set tkedit_FullPath {}
}

# Make sure the name doesn't correspond to a directory.  It doesn't have
# to be a file which exists, but it can't be a directory.
#
if {"$FullPath"!="" && [file exists $FullPath] && ![file readable $FullPath]} {
  puts stderr "tkedit: Can't open $FullPath for reading"
  exit 1
}

# Construct the editor
#
if {$Title!=""} {
  wm title . $Title
  wm iconname . $Title
} elseif {"$FullPath"!=""} {
  wm title . [file tail $FullPath]
  wm iconname . [file tail $FullPath]
} else {
  wm title . <Unnamed>
  wm iconname . <Unnamed>
}
wm minsize . 320 240
wm protocol . WM_DELETE_WINDOW FileQuit

# These bindings cause the search screen to disappear and reappear when
# the main window is iconified or deiconified.
#
bind . <Unmap> {
  if {[string compare %W .]==0} {
    if {[winfo exists .srch]} {wm withdraw .srch}
    foreach w [winfo children .] {
      if {![regexp {^\.ss[0-9]+$} $w]} continue
      wm withdraw $w
    }
  }
}
bind . <Map> {
  if {[string compare %W .]==0} {
    if {[winfo exists .srch]} {wm deiconify .srch}
    foreach w [winfo children .] {
      if {![regexp {^\.ss[0-9]+$} $w]} continue
      wm deiconify $w
    }
    after 100 {raise .; focus .t}
  }
}

# Construct the menu bar across the top of the application
#
frame .mbar -bd 2 -relief raised
pack .mbar -side top -fill x
proc MakeMenuButton i {
  set name [string toupper [string range $i 0 0]][string range $i 1 end]
  menubutton .mbar.$i -text $name -underline 0 -menu .mbar.$i.menu -pady 0
  pack .mbar.$i -side left -padx 8
}
MakeMenuButton file
MakeMenuButton edit
MakeMenuButton view
MakeMenuButton tools
MakeMenuButton help
pack .mbar -side top -fill x

if {$ReadOnly} {
  set rostate disabled
} else {
  set rostate normal
}

menu .mbar.file.menu
.mbar.file.menu add command -label Reload -command FileRevert -underline 0
.mbar.file.menu add command -label Open... -command FileOpen -underline 0
.mbar.file.menu add command -label New -command FileNew -underline 0
.mbar.file.menu add command -label Save -command FileSave -underline 0 \
     -state $rostate
.mbar.file.menu add command -label {Save as...} -command FileSaveAs -underline 5
.mbar.file.menu add command -label Shell -command ShellOut -underline 1
#.mbar.file.menu add command -label Print... -command FilePrint -underline 0
.mbar.file.menu add separator
.mbar.file.menu add command -label Quit -command FileQuit -underline 0

menu .mbar.edit.menu
.mbar.edit.menu add command -label Cut -command EditCut -underline 2 \
   -accelerator Ctrl-x -state $rostate
.mbar.edit.menu add command -label Copy -command EditCopy -underline 0 \
   -accelerator Ctrl-c
.mbar.edit.menu add command -label Paste -command EditPaste -underline 0 \
   -accelerator Ctrl-v -state $rostate
.mbar.edit.menu add command -label {Paste Quoted} -command EditPasteQuoted \
   -underline 6 -state $rostate
.mbar.edit.menu add command -label Delete -command EditDelete -underline 0 \
   -state $rostate
.mbar.edit.menu add separator
.mbar.edit.menu add command -label Search... -command EditSearch -underline 0 \
   -accelerator Ctrl-s

menu .mbar.view.menu
.mbar.view.menu add cascade -label {Font Size} -menu .mbar.view.menu.fontsize \
   -underline 0
.mbar.view.menu add cascade -label Height -menu .mbar.view.menu.height \
   -underline 0
.mbar.view.menu add cascade -label Width -menu .mbar.view.menu.width \
   -underline 0
.mbar.view.menu add cascade -label Tabs -menu .mbar.view.menu.tabs \
   -underline 0
.mbar.view.menu add cascade -label Wrap -menu .mbar.view.menu.wrap -underline 1
.mbar.view.menu add command -label Diff -command Diff -underline 0 \
   -state $rostate
.mbar.view.menu add command -label Viewer -command SeparateViewer -underline 0
.mbar.view.menu add command -label SnapShot -command SnapShot -underline 0
.mbar.view.menu add command -label Bookmarks... -command BookmarkDialog
menu .mbar.view.menu.fontsize
if {$tcl_platform(platform)=="windows"} {
  set fonts {
    {Tiny {courier 7} 0}
    {Small {courier 8} 0}
    {Normal {courier 9} 0}
    {Big {courier 11} 0}
    {Huge {courier 13} 0}
  }
  set Font {courier 9}
} elseif {$tcl_platform(os)=="Darwin"} {
  set fonts {
    {Tiny {courier 8} 0}
    {Small {courier 10} 0}
    {Normal {courier 12} 0}
    {Big {courier 14} 0}
    {Huge {courier 16} 0}
  }
  set Font {courier 12}
} else {
  set fonts {
    {Tiny {monospace 7} 0}
    {Small {monospace 8} 0}
    {Normal {monospace 9} 0}
    {Big {monospace 11} 0}
    {Huge {monospace 13} 0}
  }
  set Font {monospace 9}
}
foreach i $fonts {
  .mbar.view.menu.fontsize add radiobutton \
     -label [lindex $i 0] -variable Font -value [lindex $i 1] \
     -command [list ChangeFont [lindex $i 1] [lindex $i 0]] -underline [lindex $i 2]
}
set Font2 -adobe-helvetica-bold-r-normal-*-12-120-75-75-p-70-iso8859-1
menu .mbar.view.menu.height
foreach i {16 20 24 30 36 40 48 58 64 72} {
  .mbar.view.menu.height add radiobutton -label "$i lines" \
    -variable Height -value $i -command "ChangeHeight $i"
}
set Height 40
#if {[winfo screenheight .]<=900} {set Height 40}
menu .mbar.view.menu.width
foreach i {40 64 80 92 112 124 140 160} {
  .mbar.view.menu.width add radiobutton -label "$i chars" \
    -variable Width -value $i -command "ChangeWidth $i"
}
set Width 80
menu .mbar.view.menu.tabs
foreach i {2 3 4 8} {
  .mbar.view.menu.tabs add radiobutton -label "$i chars" \
    -variable Tabs -value $i -command "ChangeTab $i"
}
set Tabs 8
menu .mbar.view.menu.wrap
.mbar.view.menu.wrap add radiobutton -label Word -underline 0 \
  -variable Wrap -value word -command {ChangeWrap word}
.mbar.view.menu.wrap add radiobutton -label Character -underline 0 \
  -variable Wrap -value char -command {ChangeWrap char}
.mbar.view.menu.wrap add radiobutton -label None -underline 0 \
  -variable Wrap -value none -command {ChangeWrap none}
set Wrap char

menu .mbar.tools.menu
.mbar.tools.menu add command -label Shell -command ShellOut
.mbar.tools.menu add command -label Bookmarks -command BookmarkDialog
.mbar.tools.menu add command -label Search -command EditSearch
.mbar.tools.menu add command -label Quote -command {AddPrefix {> }} \
    -state $rostate
.mbar.tools.menu add command -label Unquote -command {Unquote {> }} \
    -state $rostate
.mbar.tools.menu add command -label {Wrap Paragraph} \
    -command {ParagraphWrap 72} -state $rostate


menu .mbar.help.menu
.mbar.help.menu add command -label {About this program...} \
  -command HelpAboutThisProgram

# Put a speed bar directly below the menu bar
#
# To change the contents of the speedbar, just add (or remove) entrys
# to the list on the 2nd argument to the "foreach" statement below.  The
# first field of each entry is the text to go in the speed button, the
# second field is the procedure to call when the speed button is pressed.
#
frame .sbar -bd 1  -relief raised
pack .sbar -side top -fill x

#  {Bkmrk BookmarkDialog 0}
#  {{24} {ChangeHeight 24} 0}
#  {{40} {ChangeHeight 40} 0}
#  {{64} {ChangeHeight 64} 0}
#  {Small {ChangeFont -schumacher-clean-medium-r-normal--8-80-75-75-c-50-iso8859-1 Small} 1}
#  {Normal {ChangeFont -misc-fixed-medium-r-semicondensed-*-13-120-75-75-c-60-iso8859-1 Normal} 0}

set smallBtnFont -adobe-helvetica-medium-r-normal-*-8-80-75-75-p-46-iso8859-1

foreach btn {
  {Quit FileQuit 0}
  {Save FileSave 0}
  {Copy EditCopy 0}
  {Paste EditPaste 0}
  {Diff Diff 0}
  {Search EditSearch 3}
  {80 {ChangeWidth 80} 0}
  {> {Indent 1} 0}
  {< {Unindent 1} 0}
  {Goto {GotoLine} 0}
} {
  set lbl [lindex $btn 0]
  button .sbar.b$lbl -text $lbl \
    -font $smallBtnFont \
    -command [lindex $btn 1] -underline [lindex $btn 2] -padx 10 -pady 0
  pack .sbar.b$lbl -side left
}
entry .sbar.gotoline -relief sunken -bd 1 -width 7
bind .sbar.gotoline <Return> GotoLine
pack .sbar.gotoline -side left -before .sbar.bGoto -padx 1 -pady 1
label .sbar.where -pady 0 \
  -font -adobe-helvetica-medium-r-normal-*-10-100-75-75-p-56-iso8859-1
pack .sbar.where -side right -pady 0 -anchor e
if {[info exists DesiredGoto]} {
  .sbar.gotoline insert 0 $DesiredGoto
}

if {$ReadOnly} {
  .sbar.bSave config -state disabled
  .sbar.bPaste config -state disabled
  .sbar.bDiff config -state disabled
}

# The ".mesg" frame is used to display information about the current
# file and our position within that file, at the bottom of the screen.
#
frame .mesg -bd 1 -relief raised
pack .mesg -side bottom -fill x
label .mesg.name -text $FullPath -font $Font
pack .mesg.name -side left
frame .mesg.spacer
pack .mesg.spacer -side left -fill x -expand 1
label .mesg.line -width 20 -font $Font
pack .mesg.line -side left

# The main text widget with its scroll bar.
#
scrollbar .sb -orient vertical -command {.t yview} \
  -highlightthickness 0 -bd 1 -relief raised
pack .sb -side right -fill y
text .t -font $Font -width $Width -height $Height \
   -yscrollcommand {ChangeWhere; .sb set} \
   -wrap $Wrap -bd 1 -relief raised -padx 3 -pady 2 \
   -exportselection 0
bindtags .t .t
pack .t -fill both -expand 1
.t insert end {}
#update
#update idletasks

# Change the indicator showing the top line of the screen
#
proc ChangeWhere {} {
  scan [.t index @0,0] %d w
  global Height
  .sbar.where config -text "$w..[expr $w+$Height-1]"
}

# Change the height of the editor.  The argument is the number of rows
# of text to display at one time.
#
proc ChangeHeight newheight {
  global Width Height
  .t config -height $newheight -width $Width
  set Height $newheight
  ResizeRoot
}

# Change the width of the editor.  The argument is the number of columns
# of text to display.
#
proc ChangeWidth newwidth {
  global Height Width
  .t config -height $Height -width $newwidth
  set Width $newwidth
  ResizeRoot
}

# Change the size of the root window (.) whenever the text widget
# is resized.
#
proc ResizeRoot {} {
  update idletasks
  set w [winfo reqwidth .]
  set h [winfo reqheight .]
  wm geometry . ${w}x$h
}

# Set the tab spacing
#
proc ChangeTab spacing {
  global Width Tabs
  set sp [expr int($spacing*([winfo reqwidth .t]/$Width))]
  .t config -tabs $sp
  set Tabs $spacing
}

# Change the word-wrap mode for the editor.  The argument should be
# one of:   word   char   none
#
proc ChangeWrap newmode {
  .t config -wrap $newmode
}

# Change the font used by the editor.  The argument is the name of the
# font to use.  For best results, use a constant-width font.
#
proc ChangeFont {newfont {fontname {}}} {
  global Font FontName Font2
  .t config -font $newfont
  .mesg.name config -font $newfont
  .mesg.line config -font $newfont
  .sbar.where config -font $newfont
  switch $fontname {
    Tiny -
    Small {
      set f2 -adobe-helvetica-bold-r-normal-*-10-100-75-75-p-60-iso8859-1
    }
    default {
      set f2 -adobe-helvetica-bold-r-normal-*-12-120-75-75-p-70-iso8859-1
    }
  }
  foreach w [winfo children .mbar] {
    $w config -font $f2
  }
  set Font2 $f2
  set Font $newfont
  set FontName $fontname
  ResizeRoot
}

# Attempt to launch a shell in the same directory as the file currently
# being edited.  We try to launch tkterm first, but if that fails we
# try xterm as a backup.
#
proc ShellOut {} {
  global FullPath
  cd [file dir $FullPath]
  if {[catch {exec tkterm &}]} {
    catch {exec xterm &}
  }
}

# The procedure defined below implements a generic dialog box.  The
# arguments are as follows:
#
#   position      The new dialog box is centered over the window given
#                 by this argument
#
#   title         This is the title for the dialog box
#
#   build         This procedure is called to construct the top-most
#                 panel of the dialog box.  The first argument to the
#                 procedure is the name of the frame widget which
#                 is the top panel.  Subsequent arguments are given by
#                 the "buildargs" parameter.
#
#   buildargs     This is arguments to the "build" command which come
#                 after the name of the top panel widget.
#
#   btns          This is a list of button descriptions.  Each button
#                 description consists of the name of the button and
#                 some text to be displayed beside that button.
#
# The procedure builds a model dialog box and waits for a button to be
# pressed.  When a button is pressed, the dialog box goes away and the
# procedure returns an integer which is the index of the selected button.
# The first button is numbered 0.
#
proc ModalDialogBox {pos title build buildargs btns} {
  global dialog_button
  if [winfo exists .d] {destroy .d}
  toplevel .d -class Dialog
  wm title .d $title
  wm iconname .d Dialog
  frame .d.msg -relief raised -bd 1
  $build .d.msg $buildargs
  pack .d.msg -side top -fill both -expand 1
  set cnt -1
  foreach btn $btns {
    incr cnt
    set btnname [lindex $btn 0]
    set btntext [lindex $btn 1]
    frame .d.x$cnt -relief raised -bd 1
    if [llength $btn]==3 {
      set cmd "[lindex $btn 2] .d; set dialog_button $cnt"
    } else {
      set cmd "set dialog_button $cnt"
    }
    button .d.x$cnt.btn -text $btnname -command $cmd -width 9 -underline 0
    pack .d.x$cnt.btn -side left -padx 5 -pady 5
    message .d.x$cnt.msg -text $btntext -width 10c
    pack .d.x$cnt.msg -anchor w -padx 5 -pady 5
    pack .d.x$cnt -side top -fill x
  }
  wm withdraw .d
  update idletasks
  set x [expr [winfo rootx $pos] + ([winfo width $pos]-[winfo reqwidth .d])/2]
  set y [expr [winfo rooty $pos] + ([winfo height $pos]-[winfo reqheight .d])/2]
  wm geometry .d +$x+$y
  wm deiconify .d
  set old_focus [focus]
  focus .d
  grab set .d
  bind .d <ButtonPress> {
    if {![string match .d.* %W]} {
      bell
      wm withdraw .d
      wm deiconify .d
    }
  }
  bind .d <KeyPress> {tkTraverseToMenu %W %A}
  tkwait variable dialog_button
  grab release .d
  focus $old_focus
  destroy .d
  return $dialog_button
}

#
# The following procedures are used to construct a dialog box header which
# contains an icon and a message.  The difference is in the icon.
#
proc QuestionMessage {w text} {
  label $w.icon -bitmap info
  pack $w.icon -side left -padx 15 -pady 15
  message $w.msg -text $text -width 10c
  pack $w.msg -padx 15 -pady 15 -anchor w
}
proc WarningMessage {w text} {
  label $w.icon -bitmap warning
  pack $w.icon -side left -padx 15 -pady 15
  message $w.msg -text $text -width 10c
  pack $w.msg -padx 15 -pady 15 -anchor w
}

#
# The following procedure creates a search dialog box.  Or, if the dialog
# box already exists, it raises and deiconifies it.
#
set SearchData(type) exact
set SearchData(from) {}
set SearchData(to) {}
proc EditSearch {} {
  global FullPath SearchData Font Font2
  set f1 $Font
  set f2 $Font2
  if [winfo exists .srch] {
    # wm deiconify .srch
    # raise .srch
    focus .srch.pat.e
    return
  }
  # toplevel .srch -class Dialog
  # wm title .srch [file tail $FullPath]
  # wm iconname .srch [file tail $FullPath]
  # wm withdraw .srch
  frame .srch
  pack .srch -before .sbar -side top -fill x
  frame .srch.btn
  pack .srch.btn -side right -fill x -anchor n
  foreach b {
    {replace  {Change}       1  Replace}
    {all      {Chng All}     7  ReplaceAll}
    {close    Close          0  {focus .t; destroy .srch}}
  } {
    button .srch.btn.[lindex $b 0] -text [lindex $b 1] -command [lindex $b 3] \
      -underline [lindex $b 2] -font $f1
    pack .srch.btn.[lindex $b 0] -fill x -side top -expand 1
  }
  frame .srch.btn2
  pack .srch.btn2 -side right -fill x -anchor n
  foreach b {
    {next     {Next}    0  FindNext}
    {prev     {Prev}    0  FindPrev}
  } {
    button .srch.btn2.[lindex $b 0] -text [lindex $b 1] -command [lindex $b 3] \
       -underline [lindex $b 2] -font $f1
    pack .srch.btn2.[lindex $b 0] -fill x -side top -expand 1
  }
  frame .srch.ctrl
  pack .srch.ctrl -side right -fill y -padx 5
  radiobutton .srch.ctrl.b1 -text Exact -variable SearchData(type) \
     -value exact -pady 0 -underline 0 -font $f1
  radiobutton .srch.ctrl.b2 -text {No Case} -variable SearchData(type) \
     -value nocase -pady 0 -underline 1 -font $f1
  radiobutton .srch.ctrl.b3 -text {Reg Exp} -variable SearchData(type) \
     -value regexp -pady 0 -underline 5 -font $f1
  pack .srch.ctrl.b1 .srch.ctrl.b2 .srch.ctrl.b3 -side top -anchor w
  frame .srch.pat
  pack .srch.pat -side top -padx 5 -pady 2 -fill x
  label .srch.pat.l -text {Search for:} -width 12 -anchor e -font $f1
  pack .srch.pat.l -side left
  entry .srch.pat.e -width 40 -bd 2 -relief sunken \
     -textvariable SearchData(from) -font $f1
  pack .srch.pat.e -side left -fill x
  bind .srch.pat.e <Return> FindNext
  bind .srch.pat.e <F3> FindNext
  bind .srch.pat.e <F4> FindPrev
  bind .srch.pat.e <F8> Replace
  frame .srch.to
  pack .srch.to -side top -padx 5 -pady 2 -fill x
  label .srch.to.l -text {Change to:} -width 12 -anchor e -font $f1
  pack .srch.to.l -side left
  entry .srch.to.e -width 40 -bd 2 -relief sunken \
     -textvariable SearchData(to) -font $f1
  pack .srch.to.e -side left -fill x
  bind .srch.to.e <Return> FindNext
  bind .srch.to.e <F3> FindNext
  bind .srch.to.e <F4> FindPrev
  bind .srch.to.e <F8> Replace
  update idletasks
  set y [expr {[winfo rooty .] - [winfo reqheight .srch] - 40}]
  if {$y<0} {set y 0}
  # wm geometry .srch +[winfo rootx .]+$y
  # wm deiconify .srch
  focus .srch.pat.e
}

#
# Using the pattern string located in the entry widget .srch.pat.e,
# find the next occurance of the pattern in the text widget.
#
proc FindNext {} {
  global SearchData
  set type -$SearchData(type)
  set pattern $SearchData(from)
  set count 0
  set w [.t search -count count $type -- $pattern {insert +1 chars}]
  if {"$w"==""} {
    set w [.t search -count count $type -- $pattern 1.0 insert]
  }
  if {"$w"!=""} {
    .t mark set insert $w
    textFixup
    textSel insert "insert +$count chars"
  }
  focus .t
}
proc FindPrev {} {
  global SearchData
  set type -$SearchData(type)
  set pattern $SearchData(from)
  set count 0
  set w [.t search -count count -backwards $type -- $pattern {insert -1 chars}]
  if {"$w"==""} {
    set w [.t search -count count -backwards $type -- $pattern end insert]
  }
  if {"$w"!=""} {
    .t mark set insert $w
    textFixup
    textSel insert "insert +$count chars"
  }
  focus .t
}

#
# Move to the line number found in the .sbar.gotoline entry box.
#
proc GotoLine {} {
  set x [.sbar.gotoline get]
  regsub -all {[^0-9]} $x {} x
  .t mark set insert $x.0
  textFixup
  focus .t
}

#
# If the text at the insertion cursor matches the pattern in the
# search dialog, then replace the pattern with the substitution string
# and find the next occurance of the pattern.
#
proc Replace {} {
  global Dirty SearchData
  set pattern $SearchData(from)
  set new $SearchData(to)
  set mode -$SearchData(type)
  set count 0
  set w [.t search $mode -count count -- $pattern insert {insert +2 line}]
  if {[string length $w]>0 && [.t compare insert == $w]} {
    if {$count>0} {.t delete insert "insert +$count chars"}
    .t insert insert $new
    set Dirty 1
    FindNext
    return 1
  } else {
    return 0
  }
}
proc ReplaceAll {} {
  .t mark set insert 1.0
  FindNext
  set start [.t index insert]
  while {[Replace] && [.t index insert]>$start} {}
}

#
# Create a bookmark dialog box.
#
proc BookmarkDialog {} {
  if {[winfo exists .md]} {
    wm deiconify .md
    raise .md
    return
  }
  toplevel .md -class Dialog
  global FullPath Bookmark
  wm title .md [file tail $FullPath]
  wm iconname .md [file tail $FullPath]
  set w [frame .md.top]
  pack $w -side top -fill x
  button $w.add -text {Add Bookmark} -command BookmarkAdd
  button $w.back -text {Go Back} -command BookmarkBack
  button $w.close -text {Dismiss} -command {destroy .md}
  pack $w.add $w.back $w.close -side left -fill x -expand 1
  set Bookmark(cnt) 0
  foreach bm [lsort [array names Bookmark m:*]] {
    set label [string range $bm 2 end]
    BookmarkInsertButton $label
  }
  BookmarkResizeWindow
}

# Change the size of the bookmark window (.md) to whatever size is
# requested by its contained widgets.
#
proc BookmarkResizeWindow {} {
  update idletasks
  set w [winfo reqwidth .md]
  set h [winfo reqheight .md]
  wm geometry .md ${w}x$h
}


#
# This routine moves the screen to the location of a bookmark.  The
# text associated with the bookmark is the sole argument.
#
proc BookmarkGoto lbl {
  global Bookmark
  set loc [.t search -exact -- $lbl 1.0 end]
  if {$loc!=""} {
    if {![info exists Bookmark(prev)] || [llength $Bookmark(prev)]==0} {
      set Bookmark(prev) [.t index insert]
    } else {
      set Bookmark(prev) [concat [list [.t index insert]] $Bookmark(prev)]
      if {[llength $Bookmark(prev)]} {
        set Bookmark(prev) [lrange $Bookmark(prev) 0 20]
      }
    }
    .t mark set insert $loc
    textFixup
    after idle {focus .t}
  }
}

#
# Go back to where we were.
#
proc BookmarkBack {} {
  global Bookmark
  if {![info exists Bookmark(prev)]} return
  set loc [lindex $Bookmark(prev) 0]
  set Bookmark(prev) [lrange $Bookmark(prev) 1 end]
  if {$loc==""} return
  .t mark set insert $loc
  textFixup
  after idle {focus .t}
}

#
# Insert a new bookmark button.  The label is $label
#
proc BookmarkInsertButton label {
  global Bookmark
  set f [frame .md.x$Bookmark(cnt)]
  pack $f -side top -fill x
  label $f.l -width 30 -anchor w -text $label
  pack $f.l -side left -fill y
  button $f.go -text Go -command "BookmarkGoto [list $label]" -pady 0
  pack $f.go -side left
  button $f.del -text Del -command "BookmarkDel $Bookmark(cnt) [list $label]" \
    -pady 0
  pack $f.del -side left
  incr Bookmark(cnt)
  BookmarkResizeWindow
}

#
# Add the current selection as a bookmark.
#
proc BookmarkAdd {} {
  global Bookmark
  if {[catch {.t get {insert linestart} {insert lineend}} sel]} return
  set sel [string trim $sel]
  if {[info exists Bookmark(m:$sel)]} return
  set Bookmark(m:$sel) 1
  BookmarkInsertButton $sel
}

#
# Delete a bookmark
#
proc BookmarkDel {cnt label} {
  global Bookmark
  catch {unset Bookmark(m:$label)}
  catch {destroy .md.x$cnt}
  BookmarkResizeWindow
}


#
# The following function loads the text of a file into the text widget.
#
proc FileLoad {filename} {
  . config -cursor watch
  .t delete 1.0 end
  if [file readable $filename] {
    set f [open $filename rb]
    fconfigure $f -encoding utf-8
    .t insert end [read $f]
    close $f
  }
  focus .t
  .t mark set insert 1.0
  textFixup
  . config -cursor {}
  global Dirty
  set Dirty 0
}

#
# This routine is called in response to the "File/Quit" menu action.
#
proc FileQuit {} {
  global Dirty FullPath
  while {$Dirty} {
    set response [ModalDialogBox . {Save Changes?} QuestionMessage \
      "You have made changes to the file \"$FullPath\".\
       Do you want to save those changes before exiting?" {
         {Save     {Save the changes to the file before exiting}}
         {Discard  {Don't save the changes, but exit anyway}}
         {Cancel   {Don't exit.  Return to the editor}}
         {Diff     {Show unsaved changes in a separate window}}
      }]
    if {$response==2} return
    if {$response==0} FileSave
    if {$response==1} {set Dirty 0}
    if {$response==3} Diff
  }
  destroy .
  exit
}

#
# Write the editor contents into the file FullPath.
#
# This routine is never called unless we are sure that the file has
# write permission.
#
proc FileWrite {} {
  global FullPath Dirty
  set f [open $FullPath wb]
  fconfigure $f -encoding utf-8
  puts $f [string trimright [.t get 1.0 end]]
  close $f
  set Dirty 0
}

#
# This routine is called in order to save the editor contents to its
# default filename.
#
proc FileSave {} {
  global FullPath Dirty
  if {"$FullPath"==""} {
    FileSaveAs
    return
  }
  if {[file exists $FullPath] && ![file writable $FullPath]} {
    set response [ModalDialogBox . {Can't Write} WarningMessage \
      "The file \"$FullPath\" is not writable.  What do you want to do?" {
      {{Save As...}  {Save the editor contents to a different filename}}
      {{Try Again}   {Attempt to change permissions to allow the file to be written}}
      {{Cancel}      {Don't save the contents of the editor}}
    }]
    if {$response==2} return
    if {$response==0} {FileSaveAs; break}
    catch {exec chmod +w $FullPath}
    if {![file writable $FullPath]} {
      set response [ModalDialogbox . {Still Can't Write} WarningMessage \
        "Unable to enable write permission for the file \"$FullPath\".  What now?" {
        {{Save As...}  {Save the editor contents to a different filename}}
        {{Cancel}      {Don't save the contents of the editor}}
      }]
      if {$response==1} return
      FileSaveAs
    } else {
      FileWrite
    }
  } else {
    FileWrite
  }
}

#
# This routine is called query the user for a new filename in which
# to write the contents of a file.
#
proc FileSaveAs {} {
  global FullPath
  set newname [tk_getSaveFile -parent . -title {Save As}]
  if {$newname==""} return
  set FullPath $newname
  .mesg.name config -text $FullPath
  wm title . [file tail $FullPath]
  wm iconname . [file tail $FullPath]
  FileSave
}

#
# Create a new file editor
#
proc FileNew {} {
  global argv0
  catch "exec $argv0 &"
}

#
# Open a new file
#
proc FileOpen {} {
  # set newname [GetFileName . {Open} {*}]
  set newname [tk_getOpenFile -parent . -title Open]
  if {$newname!=""} {
    global argv0
    catch {exec $argv0 texteditor $newname &}
  }
}

#
# This routine is called in response to the File/Revert menu selection
#
proc FileRevert {} {
  global FullPath Dirty
  set oldname $FullPath
  while {$Dirty} {
    set response [ModalDialogBox . {Undo Changes?} WarningMessage \
      "The contents of the editor have been modified.\
       If you reload the editor you will lose your changes!" {
        {Revert  {Go ahead and overwrite the changes in the editor}}
        {Save    {Save editor contents to a different file before reloading}}
        {Cancel  {Don't reload the editor}}
        {Diff    {Show changes in a separate window and ask again}}
      }]
    if $response==0 break
    if $response==1 FileSaveAs
    if $response==2 return
    if $response==3 Diff
  }
  set FullPath $oldname
  .mesg.name config -text $FullPath
  set top [.t index @0,0]
  FileLoad $FullPath
  .t yview [expr {int($top)}]
}

#
# This routine is called to compute and display the line information
# in the message frame.  It should be called whenever the insertion
# cursor moves.
#
proc ComputeLine {} {
  scan [.t index insert] {%d.%*d} now
  scan [.t index end] {%d.%*d} last
  incr last -1
  .mesg.line config -text "line $now of $last"
}

# A routine for dispensing the selection.  The selection is always owned
# by the window ".".  Its value is stored in the variable "Selection"
#
set Selection {}
#selection handle -selection PRIMARY . RetrieveSelection
selection handle -selection CLIPBOARD . RetrieveSelection
proc RetrieveSelection {offset max} {
  global Selection
  return [string range $Selection $offset [expr {$offset+$max}]]
}

# This routine is called whenever "." owns the selection but another
# window claims ownership.
#
proc LoseSelection {} {
  global Selection
  set Selection {}
}

# Copy the text selected in the text widget into the Selection variable,
# then claim ownership of the selection.
#
proc EditCopy {} {
  global Selection
  catch {
    set Selection [.t get sel.first sel.last]
    clipboard clear
    clipboard append $Selection
  }
}
proc EditCut {} {
  EditCopy
  EditDelete
}
proc EditPaste {} {
  if {[catch {selection get -selection CLIPBOARD -type UTF8_STRING} \
             text]==0} {
    .t insert insert $text
    textChange
  }
}
proc EditPasteQuoted {} {
  catch {
    set x [selection get]
    regsub -all "\n" $x "\n> " x
    .t insert insert "> $x"
    textChange
  }
}
proc EditDelete {} {
  catch {
    .t delete sel.first sel.last
    textChange
  }
}

#
# Create a dialog holding a snap-shot of the currently selected text.
#
proc SnapShot {} {
  set data [.t get @0,0 "@0,[winfo height .t] lineend"]
  global FullPath
  set n 0
  while {[winfo exists .ss$n]} {incr n}
  set w .ss$n
  toplevel $w
  set title "Snapshot from \"[file tail $FullPath]\""
  wm title $w $title
  wm iconname $w $title
  button $w.close -command "destroy $w" -text "Close"
  pack $w.close -side bottom -pady 4
  text $w.t -bd 1 -relief raised \
    -font [.t cget -font] \
    -height [.t cget -height] \
    -width [.t cget -width] \
    -tabs [.t cget -tabs]
  pack $w.t -side left -fill both -expand 1
  $w.t insert end $data
  $w.t config -state disabled
}

#
# Save all edit-buffer content into a temporary file.  Return the name of the
# temporary file.
#
proc SaveToTempFile {} {
  global FullPath
  set tmpname $FullPath~edit[format %08X [expr {int(rand()*0xffffffff)}]]
  set f [open $tmpname wb]
  fconfigure $f -encoding utf-8
  puts $f [string trimright [.t get 1.0 end]]
  close $f
  return $tmpname
}

#
# Use "fossil test-diff --tk" to show the difference between the current internal
# edits and the content on disk.
#
proc Diff {} {
  global FullPath
  set tmpname [SaveToTempFile]
  exec fossil test-diff --tk $FullPath $tmpname &
  after 3000 [list file delete $tmpname]
}

#
# Bring up the current editor content in a separate read-only viewer
#
proc SeparateViewer {} {
  global argv0 FullPath
  set tmpname [SaveToTempFile]
  exec $argv0 -readonly -title "[file tail $FullPath] (readonly)" -remove $tmpname &
}

# The default bindings for the text widget are woefully inadequate.
# The following section implements improved bindings.
#
bind .t <Alt-KeyPress> {tkTraverseToMenu %W %A}
bind .t <F3> FindNext
bind .t <Alt-g> FindNext
bind .t <F4> FindPrev
bind .t <F8> Replace
if {!$ReadOnly} {
  bind .t <KeyPress> {if {"%A"!=""} {.t insert insert %A; textChange}}
  bind .t <Return> {.t insert insert \n; textChange}
  bind .t <Delete> {.t delete {insert -1 char}; textChange}
  #bind .t <Delete> {.t delete insert; textChange}
  bind .t <BackSpace> {.t delete {insert -1 char}; textChange}
  bind .t <Control-h> {.t delete {insert -1 char}; textChange}
  bind .t <Control-x> EditCut
  bind .t <<Cut>> EditCut
  bind .t <Control-v> EditPaste
  bind .t <<Paste>> EditPaste
  bind .t <Control-d> {.t delete insert; textChange}
  bind .t <Control-k> {.t delete insert {insert lineend}; textChange}
}
bind .t <Control-c> EditCopy
bind .t <<Copy>> EditCopy
bind .t <Control-u> {}
bind .t <Control-w> {}
bind .t <Control-p> {textAnchor {insert -1 line}}
bind .t <Control-a> {textAnchor {insert linestart}}
bind .t <Control-e> {textAnchor {insert lineend}}
bind .t <Control-n> {textAnchor {insert +1 line}}
bind .t <Control-b> {textAnchor {insert -1 char}}
bind .t <Control-f> {textAnchor {insert +1 char}}
bind .t <Control-s> {EditSearch}
bind .t <Alt-f> {EditSearch}
bind .t <Control-g> {FindNext}
bind .t <Left> {textAnchor {insert -1 char}}
bind .t <Right> {textAnchor {insert +1 char}}
bind .t <Home> {textAnchor {insert linestart}}
bind .t <End> {textAnchor {insert lineend}}
bind .t <Up> {textAnchor {insert -1 line}}
bind .t <Down> {textAnchor {insert +1 line}}
bind .t <Prior> {textAnchor "insert -$Height lines"}
bind .t <Next> {textAnchor "insert +$Height lines"}
bind .t <Shift-Left> {textInsert {insert -1 char}; textSel tk_anchor insert}
bind .t <Shift-Right> {textInsert {insert +1 char}; textSel tk_anchor insert}
bind .t <Shift-Home> {textInsert {insert linestart}; textSel tk_anchor insert}
bind .t <Shift-End> {textInsert {insert lineend}; textSel tk_anchor insert}
bind .t <Shift-Up> {textInsert {insert -1 line}; textSel tk_anchor insert}
bind .t <Shift-Down> {textInsert {insert +1 line}; textSel tk_anchor insert}
bind .t <Shift-Prior> {textInsert "insert -$Height lines";
  textSel tk_anchor insert}
bind .t <Control-Left> {textInsert {insert -1 char}; textSel tk_anchor insert}
bind .t <Control-Right> {textInsert {insert +1 char}; textSel tk_anchor insert}
bind .t <Control-Home> {textInsert {insert linestart}; textSel tk_anchor insert}
bind .t <Control-End> {textInsert {insert lineend}; textSel tk_anchor insert}
bind .t <Control-Up> {textInsert {insert -1 line}; textSel tk_anchor insert}
bind .t <Control-Down> {textInsert {insert +1 line}; textSel tk_anchor insert}
bind .t <Control-Prior> {textInsert "insert -$Height lines";
  textSel tk_anchor insert}
bind .t <Control-Next> {
  textInsert "insert +$Height lines"
  textSel tk_anchor insert
}
bind .t <1> {.t tag remove sel 0.0 end; textAnchor @%x,%y; focus .t}
#bind .t <B1-Motion> {textInsert @%x,%y; textSel tk_anchor insert}
bind .t <B1-Motion> {B1Motion %x %y}
bind .t <ButtonRelease-1> MotorOff
bind .t <Double-1> {textSel {insert wordstart} {insert wordend}}
bind .t <Triple-1> {textSel {insert linestart} {insert lineend}}
bind .t <Shift-1> {textInsert @%x,%y; focus .t; textSel tk_anchor insert}
bind .t <Control-B1-Motion> {textSel tk_anchor @%x,%y}
bind .t <2> {.t scan mark 0 %y}
bind .t <B2-Motion> {.t scan dragto 0 %y}
bind .t <MouseWheel> {.t yview scroll [expr {-%D}] units}

#
# The next group of functions are used by the bindings.
#
proc textChange {} {textFixup; global Dirty; set Dirty 1}
#proc textFixup {} {.t yview -pickplace insert; ComputeLine }
proc textFixup {} {.t see insert; ComputeLine }
proc textSel {a b} {
  if [.t compare $a < $b] {
    .t tag remove sel 0.0 $a
    .t tag add sel $a $b
    .t tag remove sel $b end
  } else {
    .t tag remove sel 0.0 $b
    .t tag add sel $b $a
    .t tag remove sel $a end
  }
}
proc textInsert {w} {.t mark set insert $w; textFixup}
proc textAnchor {w} {.t mark set tk_anchor $w; textInsert $w}

#
# This routine does auto-motoring
#
proc B1Motion {x y} {
  set m @$x,$y
  if {$y<0} {
    set d [expr int($y/20)]
    MotorOn $d
    set m "$m $d line"
  } elseif {$y>[winfo height .t]} {
    set d +[expr int(($y-[winfo height .t])/20)+1]
    MotorOn $d
    set m "$m $d line"
  } else {
    MotorOff
    textInsert $m
    textSel tk_anchor insert
  }
}
proc MotorOn d {
  global MotorDir
  set old $MotorDir
  set MotorDir $d
  if {$old==0} {
    Motor
  }
}
proc Motor {} {
  global MotorDir
  if {$MotorDir==0} return;
  textInsert "insert $MotorDir line"
  textSel tk_anchor insert
  after 100 Motor
}
proc MotorOff {} {
  global MotorDir
  set MotorDir 0
}

# xproc:  ::canvas::_word_wrap TXT MAX-WIDTH
# title:  Do word wrapping on a line of text
#
# Do word-wrapping on $TXT.  Return the result.  Each line of text
# will contain a maximum of $MAX-WIDTH characters.  Lines are broken
# by inserting a newline followed by LEFT-MARGIN spaces.
#
proc _word_wrap {txt mw} {
  set r {}
  set mwm1 [expr {$mw-1}]
  set mwp1 [expr {$mw+1}]
  set br \n
  while {[string length $txt]>$mw} {
    if {[string index $txt $mw]==" "} {
      append r $br[string range $txt 0 $mwm1]
      set txt [string range $txt $mwp1 end]
      continue
    }
    set x [string last { } [string range $txt 0 $mw]]
    if {$x<=0} {
      append r $br[string range $txt 0 $mwm1]
      set txt [string range $txt $mw end]
    } else {
      append r $br[string range $txt 0 [expr {$x-1}]]
      set txt [string range $txt [expr {$x+1}] end]
    }
  }
  append r $br$txt
  return $r
}

# Reformat the selected paragraph
#
proc ParagraphWrap {N} {
  if {[catch {.t index sel.first} start]} return
  set x [.t get sel.first sel.last]
  regsub -all {\s+} $x { } y
  set z [_word_wrap $y $N]
  .t delete sel.first sel.last
  .t insert $start $z
  textSel $start [list $start +[string length $z] chars]
}

# Indent the selection by N spaces
#
proc Indent {N} {
  set sp { }
  for {set i 1} {$i<$N} {incr i} {append sp { }}
  AddPrefix $sp
}

# Apply the prefix given to each line of text in the selection
#
proc AddPrefix {prefix} {
  if {[catch {.t index sel.first} start]} return
  set x [.t get sel.first sel.last]
  regsub -all {\n} $x "\n$prefix" y
  .t delete sel.first sel.last
  .t insert $start $prefix$y
  textSel $start [list $start +[string length $prefix$y] chars]
}


# Unindent the selection by N spaces
#
proc Unindent {N} {
  if {[catch {.t index sel.first} start]} return
  set x [.t get sel.first sel.last]
  set sp { }
  for {set i 1} {$i<$N} {incr i} {append sp { }}
  regsub -all "\n$sp" $x \n y
  if {[string range $y 0 [expr {$N-1}]]==$sp} {
    set y [string range $y $N end]
  }
  .t delete sel.first sel.last
  .t insert $start $y
  textSel $start [list $start +[string length $y] chars]
}
proc Unquote {prefix} {
  if {[catch {.t index sel.first} start]} return
  set x [.t get sel.first sel.last]
  set N [string length $prefix]
  regsub -all "\n$prefix" $x \n y
  if {[string range $y 0 [expr {$N-1}]]==$prefix} {
    set y [string range $y $N end]
  }
  .t delete sel.first sel.last
  .t insert $start $y
  textSel $start [list $start +[string length $y] chars]
}

# tkMenuFind --
# This procedure searches the entire window hierarchy under w for
# a menubutton that isn't disabled and whose underlined character
# is "char".  It returns the name of that window, if found, or an
# empty string if no matching window was found.  If "char" is an
# empty string then the procedure returns the name of the first
# menubutton found that isn't disabled.
#
# If a third argument is provided, it is used as a classname pattern
# for the window to search for.  Be default, this pattern is
# MenuButton, meaning that this routine will find only menubuttons.
# But if you change the class pattern to "*utton", the routine will
# find the first button of any type.
#
# Arguments:
# w -				Name of window where key was typed.
# char -			Underlined character to search for;
#				may be either upper or lower case, and
#				will match either upper or lower case.

proc tkMenuFind {w char {pattern Menubutton}} {
    global tkPriv
    set char [string tolower $char]
    set action [format {
        %s {
            set char2 [string index [$child cget -text] \
		[$child cget -underline]]
	    if {([string compare $char [string tolower $char2]] == 0)
		|| ($char == "")} {
  	        if {[$child cget -state] != "disabled"} {
		    return $child
		}
	    }
	}

	default {
	    set match [tkMenuFind $child $char %s]
	    if {$match != ""} {
		return $match
	    }
	}
    } $pattern $pattern]

    foreach child [winfo child $w] {
	switch -glob [winfo class $child] $action
    }
    return {}
}

# tkTraverseToMenu --
# This procedure implements keyboard traversal of menus.  Given an
# ASCII character "char", it looks for a menubutton with that character
# underlined.  If one is found, it posts the menubutton's menu.
#
# The routine will also look for buttons to invoke.  If a button is
# found that contains the given character, then that button is invoked.tkp
#
# Arguments:
# w -				Window in which the key was typed (selects
#				a toplevel window).
# char -			Character that selects a menu.  The case
#				is ignored.  If an empty string, nothing
#				happens.

proc tkTraverseToMenu {w char} {
    if ![winfo exists $w] return
    global tkPriv
    if {$char == ""} {
	return
    }
    while {[winfo class $w] == "Menu"} {
	if {$tkPriv(postedMb) == ""} {
	    return
	}
	set w [winfo parent $w]
    }
    set w [tkMenuFind [winfo toplevel $w] $char *utton]
    if {$w != ""} {
        switch [winfo class $w] {
            Menubutton {
   	        tkMbPost $w
	        tkMenuFirstEntry [$w cget -menu]
            }

            Button {
                tkButtonInvoke $w
            }

            Checkbutton {
                tkCheckRadioInvoke $w
            }

            Radiobutton {
                tkCheckRadioInvoke $w
            }
        }
    }
}

#
# Finally, load the file into the editor and begin editing
#
FileLoad $FullPath
set idx [.t index end]
if {$idx<24.0} {
  ChangeHeight 24
} elseif {$idx<36.0} {
  ChangeHeight 36
} elseif {$idx<48.0} {
  ChangeHeight 48
}
.t mark set tk_anchor 1.0
if {$DeleteAfterLoading} {
  file delete -force $FullPath
}
if {[info exists DesiredWidth] && [string is integer -strict $DesiredWidth]} {
  ChangeWidth $DesiredWidth 
} elseif {$AutoWidth} {
  set mx 80
  foreach lx [split [.t get 1.0 end] \n] {
    if {[string length $lx]>$mx} {set mx [string length $lx]}
  }
  if {$mx>200} {set mx 200}
  ChangeWidth $mx
}
if {[info exists DesiredGoto] && [string is integer -strict $DesiredGoto]} {
  .t mark set insert $DesiredGoto.0
  textFixup
  focus .t
}